program ShowInfo;

{$IFDEF FPC}
  {$MODE Delphi}

  {$IFDEF CPUI386}
    {$DEFINE CPU386}
    {$ASMMODE INTEL}
  {$ENDIF}

  {$IFNDEF WIN32}
    {$LINKLIB c}
  {$ENDIF}
{$ELSE}
  {$APPTYPE CONSOLE}
{$ENDIF}


uses
  Classes,
  SysUtils,
  libPNG in '..\libPNG.pas';


procedure read_from_stream(png: png_structp; buffer: png_bytep; size: cardinal); cdecl;
var
  FS: TStream;
begin
  FS := TStream(png_get_io_ptr(png));

  FS.Read(buffer^, size);
end;


procedure ShowFileInfo(Filename: String);
var
  signature: array [0..7] of byte;
  png_stream: TFileStream;
  png: png_structp;
  png_info: png_infop;

  back: png_color_16p;
  gamma: Double;
  intent: Integer;

  white_x, white_y, red_x, red_y, green_x, green_y, blue_x, blue_y: Double;

  time: png_timep;
  texts: png_textp;
  texts_count: integer;
begin
  // *** reading information from filename ***
  writeln(Filename);
  writeln;
  png_stream := TFileStream.Create(Filename, fmOpenRead);

  // checking signature
  png_stream.Read(signature, 8);

  if png_check_sig(@signature, 8) = 0 then begin
    writeln('ERROR: the given file isn''t an png.');
    exit;
  end;

  // creating png struct
  png := png_create_read_struct(PNG_LIBPNG_VER_STRING, nil, nil, nil);
  if png = nil then begin
    writeln('ERROR: error creating png struct.');
    exit;
  end;

  // creating info struct
  png_info := png_create_info_struct(png);
  if png_info = nil then begin
    png_destroy_read_struct(@png, nil, nil);

    writeln('ERROR: error creating info struct');
    exit;
  end;

  // set read callback
  png_set_read_fn(png, png_stream, read_from_stream);

  // we have read the signature so skip the first 8 bytes
  png_set_sig_bytes(png, 8);

  // read informations
  png_read_info(png, png_info);

  // print generall informations
  writeln('  Width         : ', IntToStr(png_get_image_width(png, png_info)));
  writeln('  Height        : ', IntToStr(png_get_image_height(png, png_info)));
  writeln;
  
  write('  ColorType     : ');
  case png_get_color_type(png, png_info) of
    PNG_COLOR_TYPE_GRAY:        writeln('Grayscale');
    PNG_COLOR_TYPE_PALETTE:     writeln('Palette');
    PNG_COLOR_TYPE_RGB:         writeln('Color');
    PNG_COLOR_TYPE_RGB_ALPHA:   writeln('Color + Alpha');
    PNG_COLOR_TYPE_GRAY_ALPHA:  writeln('Grayscale + Alpha');
  end;
  writeln('  Channels      : ', IntToStr(png_get_channels(png, png_info)));
  writeln('  Channeldepth  : ', IntToStr(png_get_bit_depth(png, png_info)), ' bit');
  writeln;

  // print backgroundcolor
  if png_get_bKGD(png, png_info, @back) > 0 then begin
    writeln('  Background');
    writeln('    Index = ', IntToStr(back^.index));
    writeln('    Color = ', Format('(%d, %d, %d)', [back^.red, back^.green, back^.blue]));
    writeln('    Gray  = ', IntToStr(back^.gray));
  end;

  // print gamma
  if png_get_gAMA(png, png_info, @gamma) > 0 then
    writeln('  Gamma         : ', Format('%0.3f', [gamma]));

  // print sRGB
  if png_get_sRGB(png, png_info, @intent) > 0 then
    writeln('  sRGB          : ', IntToStr(intent));

  // print CIE color characteristics
  if png_get_cHRM(png, png_info, @white_x, @white_y, @red_x, @red_y, @green_x, @green_y, @blue_x, @blue_y) > 0 then begin
    writeln('  CIE');
    writeln('    white x = ', Format('%0.3f', [white_x]));
    writeln('    white y = ', Format('%0.3f', [white_y]));
    writeln('    red x   = ', Format('%0.3f', [red_x]));
    writeln('    red y   = ', Format('%0.3f', [red_y]));
    writeln('    green x = ', Format('%0.3f', [green_x]));
    writeln('    green y = ', Format('%0.3f', [green_y]));
    writeln('    blue x  = ', Format('%0.3f', [blue_x]));
    writeln('    blue y  = ', Format('%0.3f', [blue_y]));
  end;

(*
  // print CAL color characteristics
  if png_get_sCAL(png, png_info, nil, nil, nil) > 0 then begin
    writeln('  CAL       :');
    writeln('    Desc    : ', png_info^.pcal_purpose);
  end;
*)

  // print time
  if png_get_tIME(png, png_info, @time) > 0 then
    with time^ do
      writeln('  Last Modified : ', Format('%0.2d.%0.2d.%0.4d - %0.2d:%0.2d:%0.2d', [day, month, year, hour, minute, second]));

  writeln;
  write('  Interlace     : ');
  case png_get_interlace_type(png, png_info) of
    PNG_INTERLACE_NONE:  writeln('None');
    PNG_INTERLACE_ADAM7: writeln('Adam7');
  end;

  // print texts
  if png_get_text(png, png_info, @texts, @texts_count) > 0 then begin
    writeln;
    writeln('  Texts');//, IntToStr(texts_count));

    while texts_count > 0 do begin
      writeln('    ', texts^.key, ' = "', texts^.text, '"');

      inc(texts);
      dec(texts_count);
    end;
  end;

  // *** finalization input ***
  png_destroy_read_struct(@png, @png_info, nil);

  png_stream.Free;

  writeln('');
end;


begin
  if ParamCount = 0 then begin
    writeln('ShowInfo displays information about an incoming png file.');
    writeln('');
    writeln('usage:');
    writeln('  ShowInfo.exe PNGImage');
    halt;
  end;

  // *** initialization ***
  if not init_libPNG then begin
    writeln('ERROR: Couldn''t initialize libPNG.');
    halt;
  end;

  // *** version ***
  writeln('ShowInfo using libPNG (', png_get_libpng_ver(nil), ')');
  writeln('');

  ShowFileInfo(ParamStr(1));

  // not really necessary in this case
  quit_libPNG;
end.
 