﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Data.SQLite;
using System.Drawing.Imaging;
using System.Reflection;
using System.Globalization;
using System.Threading;
using System.Collections;
using System.Runtime.InteropServices;
using System.Drawing.Drawing2D;

namespace Mapv4v5Converter
{
    public enum ImgFormat
    { source = 0, png = 1, cjpg = 2, jpg = 3, unifill = 4, unknown = 5 };
    public partial class Form1 : Form
    {
        ImgFormat mapLayer = ImgFormat.source;
        ImgFormat satelliteLayer = ImgFormat.source;
        ImgFormat hybridLayer = ImgFormat.source;
        int mapJpgQuality = 75, satelliteJpgQuality = 75, hybridJpgQuality = 75;

        protected static byte[] cjpegHeader;
        protected static byte[] jpegHeader;
        bool CancelConvert = false;

        public Form1()
        {
//            Thread.CurrentThread.CurrentUICulture = new CultureInfo("fr-FR");
            InitializeComponent();
            using (Stream jpegHeaderStream = Assembly.GetExecutingAssembly().GetManifestResourceStream("Mapv4v5Converter.header128.cjpg"))
            {
                cjpegHeader = new byte[jpegHeaderStream.Length];
                jpegHeaderStream.Read(cjpegHeader, 0, (int)jpegHeaderStream.Length);
            }
            using (Stream jpegHeaderStream = Assembly.GetExecutingAssembly().GetManifestResourceStream("Mapv4v5Converter.header.jpg"))
            {
                jpegHeader = new byte[jpegHeaderStream.Length];
                jpegHeaderStream.Read(jpegHeader, 0, (int)jpegHeaderStream.Length);
            }

        }

        private void btnBrowse_Click(object sender, EventArgs e)
        {
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = Mapv4v5Converter.Resource.ofdFilter1;
            if (ofd.ShowDialog() != DialogResult.OK)
                return;

            sourceFileName.Text = ofd.FileName;
        }

        public enum mapFrmt { unknown, xGPS /*256x256*/, motionX /*256x256*/,
            v4 = 4 /*64x64*/,
            v5 = 5 /*128x128*/,
            v10 = 6 /*256x256*/,
            v10small = 7 /*128x128*/,
            v10hiRes = 8 /*256x256*/,
            v11 = 9 /*256x256*/,
            v11small = 10 /*128x128*/,
            v11hiRes = 11 /*256x256*/,
        }
        /*
iPhone модели до iPhone4 v5 (прошивка 2.2-3.1.3)
iPhone модели до iPhone4 v10 (прошивка 4.0-4.1)
iPhone модели до iPhone4 v11 (прошивка 4.2.1 и выше)
iPhone 4 / iPod Touch 4G v10 (прошивка 4.0-4.1)
iPhone 4 / iPod Touch 4G v11 (прошивка 4.2.1 и выше)
iPad v10 (прошивка 3.2-3.2.2)
iPad v11 (прошивка 4.2.1 и выше)
xGPS
MotionX GPS         */
        mapFrmt OuputFormat
        {
            get
            {
                switch (outputFormatCb.SelectedIndex)
                {
                    case 0:
                        return mapFrmt.v5;
                    case 1:
                        return mapFrmt.v10small;
                    case 2:
                        return mapFrmt.v11small;
                    case 3:
                        return mapFrmt.v10hiRes;
                    case 4:
                        return mapFrmt.v11hiRes;
                    case 5:
                        return mapFrmt.v10;
                    case 6:
                        return mapFrmt.v11;
                    case 7:
                        return mapFrmt.xGPS;
                    case 8:
                        return mapFrmt.motionX;
                }
                return mapFrmt.unknown;
            }
        }
        private void btnConvert_Click(object sender, EventArgs e)
        {
            try
            {
                if (string.IsNullOrEmpty(sourceFileName.Text))
                {
                    MessageBox.Show(Resource.selectMap, Resource.selectMapCaption,
                        MessageBoxButtons.OK, MessageBoxIcon.Information);
                    return;
                }
                using (SQLiteConnection conn4 = new SQLiteConnection("Data Source=" + sourceFileName.Text))
                {
                    conn4.Open();
                    mapFrmt inputMapFrmt =  mapFrmt.unknown;
                    using(SQLiteCommand cmd = new SQLiteCommand(
                        "SELECT Count(*) FROM sqlite_master WHERE type='table' AND name='tiles'", conn4))
                    {
                        if (((long)cmd.ExecuteScalar()) == 1)
                            inputMapFrmt = mapFrmt.xGPS;
                    }

                    if (inputMapFrmt == mapFrmt.unknown)
                    {
                    using (SQLiteCommand cmd = new SQLiteCommand(
                        "SELECT * FROM version WHERE rowid=1", conn4))
                    {
                        object o = cmd.ExecuteScalar();
                        int ver = 0;
                        if (o != null)
                            ver = (int)o;
                        switch (ver)
                        { 
                            case 4:
                                inputMapFrmt = mapFrmt.v4;
                                break;
                            case 5:
                                inputMapFrmt = mapFrmt.v5;
                                break;
                            case 10:
                            case 11:
                                inputMapFrmt = Check10Fmt(conn4);
                                break;
                            default:
                                MessageBox.Show(Resource.invalidFile,
                                    Resource.invalidFileCaption, MessageBoxButtons.OK, MessageBoxIcon.Information);
                            return;
                        }
                    }
                    }

                    if (inputMapFrmt == mapFrmt.v10)
                    {
                        MessageBox.Show(Resource.invalidFile,
                            Resource.invalidFileCaption, MessageBoxButtons.OK, MessageBoxIcon.Information);
                        return;
                    }
                    mapFrmt outputMapFrmt = OuputFormat;
                    string outputFileName;
                    if (outputMapFrmt == mapFrmt.motionX)
                    {
                        FolderBrowserDialog dlg = new FolderBrowserDialog();
                        dlg.Description = Resource.chooseDestinationCaption;
                        if (dlg.ShowDialog() == DialogResult.Cancel)
                            return;
                        outputFileName = dlg.SelectedPath;
                    }
                    else
                    {
                        SaveFileDialog sfd = new SaveFileDialog();
                        if (outputMapFrmt == mapFrmt.v5 || outputMapFrmt == mapFrmt.v10 || outputMapFrmt == mapFrmt.v10small || outputMapFrmt == mapFrmt.v10hiRes
                            || outputMapFrmt == mapFrmt.v11 || outputMapFrmt == mapFrmt.v11small || outputMapFrmt == mapFrmt.v11hiRes)
                        {
                            sfd.DefaultExt = "sqlitedb";
                            sfd.Filter = Resource.ofdFilter;
                            sfd.FileName = "MapTiles.sqlitedb";
                        }
                        else
                        {
                            sfd.DefaultExt = "db";
                            sfd.Filter = Resource.ofdxgpsFilter;
                            sfd.FileName = "xGPS_map.db";
                        }
                        string locale = null;
                        if (outputMapFrmt == mapFrmt.v10 || outputMapFrmt == mapFrmt.v10small || outputMapFrmt == mapFrmt.v10hiRes ||
                            outputMapFrmt == mapFrmt.v11 || outputMapFrmt == mapFrmt.v11small || outputMapFrmt == mapFrmt.v11hiRes)
                        {
                            SelectLocaleDlg dlg = new SelectLocaleDlg();
                            if (dlg.ShowDialog() == DialogResult.Cancel)
                                return;
                            locale = dlg.Locale;
                        }
                        sfd.Title = Resource.chooseDestinationCaption;
                        sfd.OverwritePrompt = true;
                        if (sfd.ShowDialog() == DialogResult.Cancel)
                            return;
                        outputFileName = sfd.FileName;
                        if (outputFileName == sourceFileName.Text.Trim())
                        {
                            MessageBox.Show(Resource.selectOtherFile,
                                Resource.selectOtherFileCaption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                            return;
                        }
                        File.Delete(outputFileName);
                        if (!CreateNewBank(outputFileName, outputMapFrmt, locale))
                            return;
                    }
                    progressBar1.Value = 0;
                    progressBar1.Visible = true;
                    btnConvert.Enabled = false;
                    btnCompression.Enabled = false;
                    btnBrowse.Enabled = false;
                    sourceFileName.Enabled = false;
                    chkTransferPartial.Enabled = false;
                    btnCalcStats.Enabled = false;
                    Application.DoEvents();
                    long total = 0;
                    if (inputMapFrmt == mapFrmt.xGPS)
                    {
                        using (SQLiteCommand cmd = new SQLiteCommand(
                            "SELECT max(rowid)-min(rowid) FROM tiles", conn4))
                        {
                            total = (long)cmd.ExecuteScalar();
                        }
                    }
                    else
                    {
                        using (SQLiteCommand cmd = new SQLiteCommand(
                            "SELECT max(rowid)-min(rowid) FROM images", conn4))
                        {
                            total = (long)cmd.ExecuteScalar();
                        }
                        using (SQLiteCommand cmd1 = new SQLiteCommand(
                            @"select * from sqlite_master where 
                        sql='CREATE INDEX index1 ON images (flags,zoom,x,y)' OR
                        sql='CREATE INDEX index2 on images (flags, zoom, x, y)'", conn4))
                        {
                            if (cmd1.ExecuteScalar() == null)
                            {
                                using (SQLiteCommand cmd = new SQLiteCommand(
                                    "CREATE INDEX index2 on images (flags, zoom, x, y)", conn4))
                                {
                                    cmd.ExecuteNonQuery();
                                }
                            }
                        }
                    }
                    progressBar1.Maximum = (int)total;
                    switch (outputMapFrmt)
                    { 
                        case mapFrmt.v5:
                        case mapFrmt.v10small:
                        case mapFrmt.v11small:
                            switch (inputMapFrmt)
                            { 
                                case mapFrmt.v4:
                                    Convertv4v5(outputFileName, conn4, 64);
                                    break;
                                case mapFrmt.v5:
                                case mapFrmt.v10small:
                                case mapFrmt.v11small:
                                    Convertv5v5(outputFileName, conn4, false);
                                    break;
                                case mapFrmt.xGPS:
                                    ConvertxGPSv5(outputFileName, conn4);
                                    break;
                            }
                            break;
                        case mapFrmt.v10:
                        case mapFrmt.v11:
                            switch (inputMapFrmt)
                            {
                                case mapFrmt.v4:
                                    Convertv4v10(outputFileName, conn4);
                                    break;
                                case mapFrmt.v5:
                                case mapFrmt.v10small:
                                case mapFrmt.v11small:
                                    Convertv4v5(outputFileName, conn4, 128);
                                    break;
                                case mapFrmt.xGPS:
                                    ConvertxGPSv10(outputFileName, conn4);
                                    break;
                            }
                            break;
                        case mapFrmt.v10hiRes:
                        case mapFrmt.v11hiRes:
                            switch (inputMapFrmt)
                            {
                                case mapFrmt.v4:
                                    MessageBox.Show("This conversion is not supported. Conver to iPhone v5 format, then to iPhone 4", "Not supported", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                    return;
                                case mapFrmt.v5:
                                case mapFrmt.v10small:
                                case mapFrmt.v11small:
                                    Convertv5v5(outputFileName, conn4, true);
                                    break;
                                case mapFrmt.xGPS:
                                    MessageBox.Show("This conversion is not supported. Conver to iPhone v5 format, then to iPhone 4", "Not supported", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                                    return;
                            }
                            break;
                        case mapFrmt.xGPS:
                            switch (inputMapFrmt)
                            {
                                case mapFrmt.v4:
                                    Convertv4xGPS(new XGpsWriter(outputFileName), conn4);
                                    break;
                                case mapFrmt.v5:
                                case mapFrmt.v10small:
                                case mapFrmt.v11small:
                                    Convertv5xGPS(new XGpsWriter(outputFileName), conn4);
                                    break;
                                case mapFrmt.xGPS:
                                    ConvertxGPSxGPS(new XGpsWriter(outputFileName), conn4);
                                    break;
                            }
                            break;
                        case mapFrmt.motionX:
                            switch (inputMapFrmt)
                            {
                                case mapFrmt.v4:
                                    Convertv4xGPS(new MotionXWriter(outputFileName), conn4);
                                    break;
                                case mapFrmt.v5:
                                case mapFrmt.v10small:
                                case mapFrmt.v11small:
                                    Convertv5xGPS(new MotionXWriter(outputFileName), conn4);
                                    break;
                                case mapFrmt.xGPS:
                                    ConvertxGPSxGPS(new MotionXWriter(outputFileName), conn4);
                                    break;
                            }
                            break;
                    }
                }
                MessageBox.Show(Resource.finished, Resource.finishedCaption,
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
            }

            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), Resource.errorCaption, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
            finally
            {
                btnConvert.Enabled = true;
                progressBar1.Visible = false;
                btnBrowse.Enabled = true;
                sourceFileName.Enabled = true;
                chkTransferPartial.Enabled = true;
                btnCompression.Enabled = true;
                btnCalcStats.Enabled = true;
            }
        }


        //Не умеет отличать mapFrmt.v10 от mapFrmt.v10hiRes, всегда возвращает mapFrmt.v10
        public static mapFrmt Check10Fmt(SQLiteConnection conn4)
        {
            using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT data FROM images", conn4))
            {
                using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                {
                    while (rdr.Read())
                    {
                        byte[] img = (byte[])rdr["data"];
                        if (img.Length < 4)
                            continue;
                        Bitmap newImg = ConvertFromCjpg(img);
                        if (newImg == null)
                            continue;
                        if (newImg.Width == 128)
                            return mapFrmt.v10small;
                        else
                            return mapFrmt.v10;
                    }
                }
            }
            return mapFrmt.v10small;
        }

        private void ConvertxGPSv10(string newFileName, SQLiteConnection conn4)
        {
            int current = 0;
            using (SQLiteConnection conn5 = new SQLiteConnection("Data Source=" + newFileName))
            {
                conn5.Open();
                SQLiteTransaction tran = conn5.BeginTransaction();
                using (SQLiteCommand insCmd = new SQLiteCommand(
                    @"INSERT INTO images (zoom, x, y, flags, length, data)
                        VALUES (@zoom, @x, @y, @flags, @length, @data)", conn5))
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM tiles", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            if (current % 200 == 0)
                            {
                                progressBar1.Value = current;
                                Application.DoEvents();
                            }
                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["img"]);
                            if (newImg != null)
                            {
                                ImgFormat format;
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    format = ImgFormat.cjpg;
                                else
                                    format = ImgFormat.png;

                                int flags;
                                switch ((long)rdr["type"])
                                {
                                    case 0:
                                        flags = 2;
                                        break;
                                    case 1:
                                        flags = 6;
                                        break;
                                    default:
                                        continue;
                                }

                                SaveBmpv5(newImg, format, (int)(long)rdr["x"], (int)(long)rdr["y"], flags, 18 - (int)(long)rdr["zoom"], insCmd);
                            }
                        }
                    }
                }
                tran.Commit();
            }
        }

        private void ConvertxGPSxGPS(IMapWriter mapWriter, SQLiteConnection conn4)
        {
            try
            {
                int current = 0;
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM tiles", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            if (current % 200 == 0)
                            {
                                progressBar1.Value = current;
                                Application.DoEvents();
                            }
                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["img"]);
                            if (newImg != null)
                            {
                                ImgFormat format;
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    format = ImgFormat.jpg;
                                else
                                    format = ImgFormat.png;

                                int type = 0;
                                if ((long)rdr["type"] == 0)
                                    type = 2;
                                else
                                    type = 6;
                                mapWriter.SaveTile(newImg, format, (int)(long)rdr["x"], (int)(long)rdr["y"],
                                    type, 18 - (int)(long)rdr["zoom"]);
                            }

                        }
                    }
                }
            }
            finally
            {
                mapWriter.Commit();
            }
        }

        //Склеиваем 2х2
        private void Convertv5xGPS(IMapWriter mapWriter, SQLiteConnection conn4)
        {
            try
            {
                int current = 0;
                    //Переносим только карту
                    using (SQLiteCommand selCmd = new SQLiteCommand(
                        "SELECT * FROM images WHERE flags=2 ORDER BY flags, zoom, x, y", conn4))
                    {
                        using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                        {
                            int curX = -1, curZoom = -1, curFlags = -1;
                            //                                Bitmap curBmp = null;
                            //                              int addedTiles = 0;
                            Dictionary<Point, TileData> tiles = new Dictionary<Point, TileData>();
                            while (rdr.Read() && !CancelConvert)
                            {
                                current++;
                                Point newPoint = new Point((int)rdr["x"] / 2, (int)rdr["y"] / 2);
                                if (newPoint.X != curX
                                    || (int)rdr["flags"] != curFlags || (int)rdr["zoom"] != curZoom)
                                {
                                    foreach (KeyValuePair<Point, TileData> tile in tiles)
                                    {
                                        if (tile.Value.addedTiles == 4 || chkTransferPartial.Checked)
                                        {
                                            mapWriter.SaveTile(tile.Value.image, tile.Value.format, tile.Key.X, tile.Key.Y, 2, curZoom);
                                        }
                                    }
                                    tiles.Clear();
                                    progressBar1.Value = current;

                                    Application.DoEvents();
                                }
                                curX = newPoint.X;
                                curZoom = (int)rdr["zoom"];
                                curFlags = (int)rdr["flags"];
                                TileData data;
                                if (!tiles.TryGetValue(newPoint, out data))
                                {
                                    data = new TileData(256);
                                    tiles[newPoint] = data;
                                }

                                Bitmap newImg = ConvertFromCjpg((byte[])rdr["data"]);
                                //                                    Graphics g = Graphics.FromImage(data.image);
                                if (newImg != null)
                                {
                                    data.g.DrawImage(newImg, 128 * ((int)rdr["x"] % 2),
                                        128 * ((int)rdr["y"] % 2),
                                        128, 128);
                                    if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                        data.format = ImgFormat.cjpg;
                                    else
                                        data.format = ImgFormat.png;

                                    data.addedTiles++;
                                }

                            }
                        }
                    }
            }
            finally
            {
                mapWriter.Commit();
            }
        }

        //Склеиваем 4х4
        private void Convertv4xGPS(IMapWriter mapWriter, SQLiteConnection conn4)
        {
            try
            {
                int current = 0;
                //Переносим только карту
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM images WHERE flags=2 ORDER BY flags, zoom, x, y", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        int curX = -1, curZoom = -1, curFlags = -1;
                        //                                Bitmap curBmp = null;
                        //                              int addedTiles = 0;
                        Dictionary<Point, TileData> tiles = new Dictionary<Point, TileData>();
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            Point newPoint = new Point((int)rdr["x"] / 4, (int)rdr["y"] / 4);
                            if (newPoint.X != curX
                                || (int)rdr["flags"] != curFlags || (int)rdr["zoom"] != curZoom)
                            {
                                foreach (KeyValuePair<Point, TileData> tile in tiles)
                                {
                                    if (tile.Value.addedTiles == 16 || chkTransferPartial.Checked)
                                    {
                                        mapWriter.SaveTile(tile.Value.image, tile.Value.format, tile.Key.X, tile.Key.Y, 2, curZoom);
                                    }
                                }
                                tiles.Clear();
                                progressBar1.Value = current;

                                Application.DoEvents();
                            }
                            curX = newPoint.X;
                            curZoom = (int)rdr["zoom"];
                            curFlags = (int)rdr["flags"];
                            TileData data;
                            if (!tiles.TryGetValue(newPoint, out data))
                            {
                                data = new TileData(256);
                                tiles[newPoint] = data;
                            }

                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["data"]);
                            //                                    Graphics g = Graphics.FromImage(data.image);
                            if (newImg != null)
                            {
                                data.g.DrawImage(newImg, 64 * ((int)rdr["x"] % 4),
                                    64 * ((int)rdr["y"] % 4),
                                    64, 64);
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    data.format = ImgFormat.cjpg;
                                else
                                    data.format = ImgFormat.png;

                                data.addedTiles++;
                            }

                        }
                    }
                }
            }
            finally
            {
                mapWriter.Commit();
            }
        }

        //Разрезание на 2х2
        private void ConvertxGPSv5(string newFileName, SQLiteConnection conn4)
        {
            int current = 0;
            using (SQLiteConnection conn5 = new SQLiteConnection("Data Source=" + newFileName))
            {
                conn5.Open();
                SQLiteTransaction tran = conn5.BeginTransaction();
                using (SQLiteCommand insCmd = new SQLiteCommand(
                    @"INSERT INTO images (zoom, x, y, flags, length, data)
                        VALUES (@zoom, @x, @y, @flags, @length, @data)", conn5))
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM tiles", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            if (current % 200 == 0)
                            {
                                progressBar1.Value = current;
                                Application.DoEvents();
                            }
                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["img"]);
                            if (newImg != null)
                            {
                                ImgFormat format;
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    format = ImgFormat.cjpg;
                                else
                                    format = ImgFormat.png;
                                int flags;
                                switch ((long)rdr["type"])
                                { 
                                    case 0:
                                        flags = 2;
                                        break;
                                    case 1:
                                        flags = 6;
                                        break;
                                    default:
                                        continue;

                                }
                                for (int x = 0; x < 2; x++)
                                    for (int y = 0; y < 2; y++)
                                    {
                                        Bitmap bmp = newImg.Clone(new Rectangle(x * 128, y * 128, 128, 128), newImg.PixelFormat);
                                        SaveBmpv5(bmp, format, (int)(long)rdr["x"] * 2 + x, (int)(long)rdr["y"] * 2 + y,
                                            flags, 18 - (int)(long)rdr["zoom"], insCmd);
                                    }
                            }

                        }
                    }
                }
                tran.Commit();
            }
        }

        private void Convertv5v5(string newFileName, SQLiteConnection conn4, bool stretch)
        {
            int current = 0;
            using (SQLiteConnection conn5 = new SQLiteConnection("Data Source=" + newFileName))
            {
                conn5.Open();
                SQLiteTransaction tran = conn5.BeginTransaction();
                using (SQLiteCommand insCmd = new SQLiteCommand(
                    @"INSERT INTO images (zoom, x, y, flags, length, data)
                        VALUES (@zoom, @x, @y, @flags, @length, @data)", conn5))
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM images", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            if (current % 200 == 0)
                            {
                                progressBar1.Value = current;
                                Application.DoEvents();
                            }
                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["data"]);
                            if (newImg != null)
                            {
                                ImgFormat format;
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    format = ImgFormat.cjpg;
                                else
                                    format = ImgFormat.png;

                                if (!stretch)
                                    SaveBmpv5(newImg, format, (int)rdr["x"], (int)rdr["y"], (int)rdr["flags"], (int)rdr["zoom"], insCmd);
                                else 
                                {
                                    Bitmap stretched = new Bitmap(256, 256);
                                    Graphics g = Graphics.FromImage(stretched);
                                    g.InterpolationMode = InterpolationMode.NearestNeighbor;
                                    g.PixelOffsetMode = PixelOffsetMode.HighQuality;
                                    g.DrawImage(newImg, 0, 0, 256, 256);
                                    SaveBmpv5(stretched, format, (int)rdr["x"], (int)rdr["y"], (int)rdr["flags"], (int)rdr["zoom"] + 1, insCmd);
                                }
                            }

                        }
                    }
                }
                tran.Commit();
            }
        }

        private void Convertv4v5(string newFileName, SQLiteConnection conn4, int size)
        {
            int current = 0;
            using (SQLiteConnection conn5 = new SQLiteConnection("Data Source=" + newFileName))
            {
                conn5.Open();
                SQLiteTransaction tran = conn5.BeginTransaction();
                using (SQLiteCommand insCmd = new SQLiteCommand(
                    @"INSERT INTO images (zoom, x, y, flags, length, data)
                        VALUES (@zoom, @x, @y, @flags, @length, @data)", conn5))
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM images ORDER BY flags, zoom, x, y", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        int curX = -1, curZoom = -1, curFlags = -1;
                        //                                Bitmap curBmp = null;
                        //                              int addedTiles = 0;
                        Dictionary<Point, TileData> tiles = new Dictionary<Point, TileData>();
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            Point newPoint = new Point((int)rdr["x"] / 2, (int)rdr["y"] / 2);
                            if (newPoint.X != curX
                                || (int)rdr["flags"] != curFlags || (int)rdr["zoom"] != curZoom)
                            {
                                foreach (KeyValuePair<Point, TileData> tile in tiles)
                                {
                                    if (tile.Value.addedTiles == 4 || chkTransferPartial.Checked)
                                    {
                                        SaveBmpv5(tile.Value.image, tile.Value.format, tile.Key.X, tile.Key.Y, curFlags, curZoom, insCmd);
                                    }
                                }
                                tiles.Clear();
                                progressBar1.Value = current;

                                Application.DoEvents();
                            }
                            curX = newPoint.X;
                            curZoom = (int)rdr["zoom"];
                            curFlags = (int)rdr["flags"];
                            TileData data;
                            if (!tiles.TryGetValue(newPoint, out data))
                            {
                                data = new TileData(size * 2);
                                tiles[newPoint] = data;
                            }

                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["data"]);
                            //                                    Graphics g = Graphics.FromImage(data.image);
                            if (newImg != null)
                            {
                                data.g.DrawImage(newImg, size * ((int)rdr["x"] % 2),
                                    size * ((int)rdr["y"] % 2),
                                    size, size);
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    data.format = ImgFormat.cjpg;
                                else
                                    data.format = ImgFormat.png;

                                data.addedTiles++;
                            }

                        }
                    }
                }
                tran.Commit();
            }
        }
        //Конвертация 64->256
        private void Convertv4v10(string newFileName, SQLiteConnection conn4)
        {
            int current = 0;
            using (SQLiteConnection conn5 = new SQLiteConnection("Data Source=" + newFileName))
            {
                conn5.Open();
                SQLiteTransaction tran = conn5.BeginTransaction();
                using (SQLiteCommand insCmd = new SQLiteCommand(
                    @"INSERT INTO images (zoom, x, y, flags, length, data)
                        VALUES (@zoom, @x, @y, @flags, @length, @data)", conn5))
                using (SQLiteCommand selCmd = new SQLiteCommand(
                    "SELECT * FROM images ORDER BY flags, zoom, x, y", conn4))
                {
                    using (SQLiteDataReader rdr = selCmd.ExecuteReader())
                    {
                        int curX = -1, curZoom = -1, curFlags = -1;
                        //                                Bitmap curBmp = null;
                        //                              int addedTiles = 0;
                        Dictionary<Point, TileData> tiles = new Dictionary<Point, TileData>();
                        while (rdr.Read() && !CancelConvert)
                        {
                            current++;
                            Point newPoint = new Point((int)rdr["x"] / 4, (int)rdr["y"] / 4);
                            if (newPoint.X != curX
                                || (int)rdr["flags"] != curFlags || (int)rdr["zoom"] != curZoom)
                            {
                                foreach (KeyValuePair<Point, TileData> tile in tiles)
                                {
                                    if (tile.Value.addedTiles == 16 || chkTransferPartial.Checked)
                                    {
                                        SaveBmpv5(tile.Value.image, tile.Value.format, tile.Key.X, tile.Key.Y, curFlags, curZoom, insCmd);
                                    }
                                }
                                tiles.Clear();
                                progressBar1.Value = current;

                                Application.DoEvents();
                            }
                            curX = newPoint.X;
                            curZoom = (int)rdr["zoom"];
                            curFlags = (int)rdr["flags"];
                            TileData data;
                            if (!tiles.TryGetValue(newPoint, out data))
                            {
                                data = new TileData(256);
                                tiles[newPoint] = data;
                            }

                            Bitmap newImg = ConvertFromCjpg((byte[])rdr["data"]);
                            //                                    Graphics g = Graphics.FromImage(data.image);
                            if (newImg != null)
                            {
                                data.g.DrawImage(newImg, 64 * ((int)rdr["x"] % 4),
                                    64 * ((int)rdr["y"] % 4),
                                    64, 64);
                                if (newImg.RawFormat.Equals(ImageFormat.Jpeg))
                                    data.format = ImgFormat.cjpg;
                                else
                                    data.format = ImgFormat.png;

                                data.addedTiles++;
                            }

                        }
                    }
                }
                tran.Commit();
            }
        }

        
        public static Bitmap ConvertFromCjpg(byte[] img)
        {
            if (img.Length == 3)
            {
                Bitmap ret = new Bitmap(64, 64);
                using (Graphics g = Graphics.FromImage(ret))
                {
                    g.FillRectangle(new SolidBrush(Color.FromArgb(img[0], img[1], img[2])), 0, 0, 64, 64);
                }
                return ret;
            }
            else if (img.Length < 4)
                return null;
            //Проверяем CJPG
            else if (img[0] == 67 && img[1] == 74 && img[2] == 80 && img[3] == 71
                /*Encoding.ASCII.GetString(img, 0, 4) == "CJPG"*/)
            {
                byte[] imgTmp = new byte[jpegHeader.Length + img.Length];
                jpegHeader.CopyTo(imgTmp, 0);
                Array.ConstrainedCopy(img, 11, imgTmp, jpegHeader.Length, img.Length - 11);
                AdjustJpegheader(imgTmp, img);
                img = imgTmp;
            }
            return (Bitmap)Image.FromStream(new MemoryStream(img), false, false);
        }

        private static void AdjustJpegheader(byte[] Jpg, byte[] Cjpg)
        {
            int i, offset = 0;
            int quality = Cjpg[9];
            for (i = 2; i < Jpg.Length - 1; )
            {
                i += 2;
                offset = Jpg[i] * 0x100 + Jpg[i + 1];
                if (Jpg[i - 1] == 0xda)
                    break;
                else if (Jpg[i - 1] == 0xc0)
                {
                    //Высота
                    Jpg[i + 3] = Cjpg[5];
                    Jpg[i + 4] = Cjpg[6];
                    //Ширина
                    Jpg[i + 5] = Cjpg[7];
                    Jpg[i + 6] = Cjpg[8];
                }
                else if (Jpg[i - 1] == 0xdb && quality != 75)
                {
                    bool allOnes = false;
                    int sf = 0;
                    if (quality == 100)
                        allOnes = true;
                    else if (quality > 50)
                        sf = 200 - 2 * quality;
                    else
                        sf = 5000 / quality;

                    byte[] table = null;
                    if ((Jpg[i + 2] & 0xf) == 0)
                        table = std_luminance_quant_tbl;
                    else
                        table = std_chrominance_quant_tbl;
                    int pos = i + 3;
                    for (int j = 0; j < 64; j++)
                    {
                        if (allOnes)
                            Jpg[pos + j] = 1;
                        else
                        {
                            int k = (int)Math.Round((double)table[j] * sf / 100);
                            if (k < 256)
                                Jpg[pos + j] = (byte)k;
                            else
                                Jpg[pos + j] = 0xff;
                        }
                    }
                }
                i += offset;
            }
        }

        public static byte[] std_luminance_quant_tbl = {
  16,  11,  12,  14,  12,  10,  16,  14,
  13,  14,  18,  17,  16,  19,  24,  40,
  26,  24,  22,  22,  24,  49,  35,  37,
  29,  40,  58,  51,  61,  60,  57,  51,
  56,  55,  64,  72,  92,  78,  64,  68,
  87,  69,  55,  56,  80, 109,  81,  87,
  95,  98, 103, 104, 103,  62,  77, 113,
 121, 112, 100, 120,  92, 101, 103,  99
};

        public static byte[] std_chrominance_quant_tbl = {
  17,  18,  18,  24,  21,  24,  47,  26,
  26,  47,  99,  66,  56,  66,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99
};



        private void SaveBmpxGPS(Bitmap curBmp, ImgFormat curImgFormat, long curX, long curY, long type, long zoom, SQLiteCommand insCmd)
        {
            insCmd.Parameters.Clear();
            insCmd.Parameters.AddWithValue("@x", curX);
            insCmd.Parameters.AddWithValue("@y", curY);
            insCmd.Parameters.AddWithValue("@type", type);
            insCmd.Parameters.AddWithValue("@zoom", zoom);
            byte[] data = ConvertToCimg(curBmp, ImgFormat.png, 75, true);
            insCmd.Parameters.AddWithValue("@img", data);
            insCmd.ExecuteNonQuery();
        }

        private void SaveBmpv5(Bitmap curBmp, ImgFormat curImgFormat, int curX, int curY, int flags, int zoom, SQLiteCommand insCmd)
        {
            insCmd.Parameters.Clear();
            insCmd.Parameters.AddWithValue("@x", curX);
            insCmd.Parameters.AddWithValue("@y", curY);
            insCmd.Parameters.AddWithValue("@flags", flags);
            insCmd.Parameters.AddWithValue("@zoom", zoom);
            ImgFormat fmt;
            int quality;
            switch (flags)
            {
                case 2:
                    fmt = mapLayer;
                    quality = mapJpgQuality;
                    break;
                case 3:
                    fmt = satelliteLayer;
                    quality = satelliteJpgQuality;
                    break;
                case 6:
                default:
                    fmt = hybridLayer;
                    quality = hybridJpgQuality;
                    break;
            }
            if (fmt == ImgFormat.source)
                fmt = curImgFormat;
            byte[] data = ConvertToCimg(curBmp, fmt, quality);
            insCmd.Parameters.AddWithValue("@length", data.Length);
            insCmd.Parameters.AddWithValue("@data", data);
            insCmd.ExecuteNonQuery();
        }

        /// <summary>
        /// Конвертируем из обычного jpeg в cjpeg
        /// </summary>
        /// <param name="imgData">Массив данных изображения jpeg</param>
        /// <returns>Массив данных изображения cjpeg</returns>
        private static byte[] ProduceCjpg(byte[] imgData, int quality, int width)
        {
            ///Структура заголовка jpg файла следующая:
            ///ff d8 ff код_блока ст_байт_длины_блока мл_байт_длины блока ....
            ///.... ff da ст_байт_длины_блока мл_байт_длины блока ... данные изображения
            int i, offset = 0;
            for (i = 2; i < imgData.Length - 1; )
            {
                i += 2;
                offset = imgData[i] * 0x100 + imgData[i + 1];
                if (imgData[i - 1] == 0xda)
                    break;
                i += offset;
            }
            i += offset;
/*            byte[] outa = new byte[i];
            Array.ConstrainedCopy(imgData, 0, outa, 0, i);
            File.WriteAllBytes(@"C:\head.jpg", outa);
  */
            byte[] ret = new byte[imgData.Length - i + cjpegHeader.Length];
            cjpegHeader.CopyTo(ret, 0);
            ret[9] = (byte)quality;
            //Высота
            ret[5] = (byte)((width & 0xff00) >> 8);
            ret[6] = (byte)(width & 0xff); 
            //Ширина
            ret[7] = (byte)((width & 0xff00) >> 8);
            ret[8] = (byte)(width & 0xff); 

            Array.ConstrainedCopy(imgData, i, ret, cjpegHeader.Length, imgData.Length - i);
            return ret;
        }
        public static byte[] ConvertToCimg(Bitmap img, ImgFormat ImgFormat, int quality)
        {
            return ConvertToCimg(img, ImgFormat, quality, false);
        }

        [System.Runtime.InteropServices.DllImport("gdi32.dll")]
        public static extern bool DeleteObject(IntPtr hObject);


        public static byte[] ConvertToCimg(Bitmap img, ImgFormat ImgFormat, int quality, bool suppressUnifill)
        {
            Color uniColor;
            if (!suppressUnifill && IsUniFill(img, out uniColor))
            {
                return new byte[] { uniColor.R, uniColor.G, uniColor.B };
            }
            //Определяем формат
            switch (ImgFormat)
            { 
                default:
                case ImgFormat.png:
                    //Пытаемся преобразовать в 256 цветов
                    switch (img.PixelFormat)
                    {
                        case PixelFormat.Format24bppRgb:
                        case PixelFormat.Format32bppArgb:
                        case PixelFormat.Format32bppRgb:
                            img = Palletize(img);
                            break;
                    }

                    //Вообще, не любой формат можно подавать на вход png энкодеру. Но мы верим, что ничего не сломается.
                    MemoryStream ms001 = new MemoryStream();
                    img.Save(ms001, ImageFormat.Png);
                    byte[] imgData1 = ms001.ToArray();
                    ms001.Close();
                    return imgData1;

                case ImgFormat.cjpg:
                case ImgFormat.jpg:
                    ImageCodecInfo codec;
                    EncoderParameters myEncoderParameters;
                    codec = GetEncoderInfo("image/jpeg");
                    myEncoderParameters = new EncoderParameters(1);
                    myEncoderParameters.Param[0] = new EncoderParameter(
                        System.Drawing.Imaging.Encoder.Quality, quality);

                    MemoryStream ms00 = new MemoryStream();
                    img.Save(ms00, codec, myEncoderParameters);

                    byte[] imgData = ms00.ToArray();
                    ms00.Close();
                    if (ImgFormat == ImgFormat.cjpg)
                        imgData = ProduceCjpg(imgData, quality, img.Width);
                    return imgData;
            }
        }

        //Вообще-то 256-цветный png не всегда занимает меньше места, чем truecolor-ный. Но очень часто.
        unsafe private static Bitmap Palletize(Bitmap src)
        {
            Bitmap dest = new Bitmap(src.Width, src.Height, PixelFormat.Format8bppIndexed);
            Hashtable palette = new Hashtable(256);
            BitmapData srcData = src.LockBits(new Rectangle(0, 0, src.Width, src.Height), ImageLockMode.ReadOnly, src.PixelFormat);
            int bpp = 0;
            switch (src.PixelFormat)
            {
                case PixelFormat.Format24bppRgb:
                    bpp = 3;
                    break;
                case PixelFormat.Format32bppArgb:
                case PixelFormat.Format32bppRgb:
                    bpp = 4;
                    break;
                default:
                    throw new NotSupportedException();
            }
            BitmapData destData = dest.LockBits(new Rectangle(0, 0, dest.Width, dest.Height), ImageLockMode.WriteOnly, PixelFormat.Format8bppIndexed);
            bool overflow = false;
            try
            {
                byte* destPtr = (byte*)destData.Scan0.ToPointer();
                byte* srcPtr = (byte*)srcData.Scan0.ToPointer();
                int totalColors = 0;
                for (int y = 0; y < srcData.Height; y++)
                    for (int x = 0; x < srcData.Width; x++)
                    {
                        Color color = Color.FromArgb(*(srcPtr + srcData.Stride * y + x * bpp + 2),
                            *(srcPtr + srcData.Stride * y + x * bpp + 1),
                            *(srcPtr + srcData.Stride * y + x * bpp));

                        int idx = 0;
                        if (palette.ContainsKey(color))
                            idx = (int)palette[color];
                        else
                        {
                            if (totalColors == 256)
                            {
                                overflow = true;
                                break;
                            }
                            palette.Add(color, totalColors);
                            idx = totalColors;
                            totalColors++;
                        }
                        *(destPtr + destData.Stride * y + x) = (byte)idx;
                    }
            }
            finally
            {
                if (srcData != null)
                    src.UnlockBits(srcData);
                if (destData != null)
                    dest.UnlockBits(destData);
            }
            if (overflow)
                dest = src;
            else
            {
                ColorPalette ncp = dest.Palette;
                foreach (DictionaryEntry de in palette)
                    ncp.Entries[(int)de.Value] = (Color)de.Key;
                dest.Palette = ncp;
            }
            return dest;
        }

        private static ImageCodecInfo GetEncoderInfo(String mimeType)
        {
            int j;
            ImageCodecInfo[] encoders;
            encoders = ImageCodecInfo.GetImageEncoders();
            for (j = 0; j < encoders.Length; ++j)
            {
                if (encoders[j].MimeType == mimeType)
                    return encoders[j];
            }
            return null;
        }


        private static bool IsUniFill(Bitmap smallTile, out Color uniColor)
        {
            int Width = smallTile.Width;
            int Height = smallTile.Height;
            if (Width == 0 || Height == 0)
            {
                uniColor = Color.Black;
                return false;
            }
            PixelFormat fmt = PixelFormat.Format32bppRgb;
            switch (smallTile.PixelFormat)
            { 
                case PixelFormat.Format32bppArgb:
                case PixelFormat.Format32bppPArgb:
                case PixelFormat.Format32bppRgb:
                    fmt = smallTile.PixelFormat;
                    break;
                default:
                    break;
            }
            BitmapData data = smallTile.LockBits(new Rectangle(0, 0, Width, Height),
                ImageLockMode.ReadOnly, fmt);
            try
            {
                unsafe
                {
                    byte* p = (byte*)data.Scan0;
                    int uc = ((int*)p)[0];
                    uniColor = Color.FromArgb(uc);
                    int nOffset = data.Stride - Width * 4;
                    for (int y = 0; y < Height; ++y)
                    {
                        for (int x = 0; x < Width; ++x)
                        {
                            if (((int*)p)[0] != uc)
                                return false;
                            p += 4;
                        }
                        p += nOffset;
                    }

                }

                return true;
            }
            finally
            {
                smallTile.UnlockBits(data);
            }
        }

        internal static bool CreateNewBank(string p, mapFrmt outputFrmt, string locale)
        {
            try
            {
                using (SQLiteConnection sql = new SQLiteConnection("Data Source=" + p))
                {
                    sql.Open();

                    using (SQLiteCommand cmdD1 = sql.CreateCommand())
                    {
                        switch (outputFrmt)
                        {
                            case mapFrmt.v5:
                                cmdD1.CommandText = "CREATE TABLE images(zoom int, x int, y int, flags int, length int, data blob)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE INDEX index1 ON images (flags,zoom,x,y)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE TABLE version(version int)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "INSERT INTO version VALUES (5)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "INSERT INTO version VALUES (0)";
                                cmdD1.ExecuteNonQuery();
                                break;
                            case mapFrmt.v10hiRes:
                            case mapFrmt.v10:
                            case mapFrmt.v10small:
                            case mapFrmt.v11hiRes:
                            case mapFrmt.v11:
                            case mapFrmt.v11small:
                                cmdD1.CommandText = "CREATE TABLE images(zoom int, x int, y int, flags int, length int, data blob)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE INDEX index1 ON images (flags,zoom,x,y)";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE TABLE version(version int, locale text)";
                                cmdD1.ExecuteNonQuery();
                                if (outputFrmt == mapFrmt.v10hiRes || outputFrmt == mapFrmt.v10 || outputFrmt == mapFrmt.v10small)
                                    cmdD1.CommandText = string.Format("INSERT INTO version (version, locale) VALUES (10, '{0}')", locale);
                                else
                                    cmdD1.CommandText = string.Format("INSERT INTO version (version, locale) VALUES (11, '{0}')", locale);
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "INSERT INTO version (version, locale) VALUES (0, NULL)";
                                cmdD1.ExecuteNonQuery();
                                break;
                            case mapFrmt.xGPS:
                                cmdD1.CommandText = "CREATE TABLE tiles (x INTEGER, y INTEGER,zoom INTEGER,type INTEGER, img BLOB,PRIMARY KEY(x,y,zoom,type))";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE TABLE maps (id INTEGER, name TEXT, zoom TEXT, type INTEGER, PRIMARY KEY(id))";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE TABLE map_regions (regionsid INTEGER, mapid INTEGER, PRIMARY KEY(regionsid))";
                                cmdD1.ExecuteNonQuery();
                                cmdD1.CommandText = "CREATE TABLE regions_points (regionsid INTEGER, lat REAL,lon REAL, pos INTEGER, PRIMARY KEY(regionsid,lat,lon))";
                                cmdD1.ExecuteNonQuery();
                                break;
                        }
                    }
                    sql.Close();
                }
            }
            catch (Exception e)
            {
                MessageBox.Show(e.Message, Resource.errorCaption, MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }
            return true;
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            toolTip1.SetToolTip(chkTransferPartial, Resource.checkBoxTooltip);
            toolTip1.SetToolTip(btnConvert, Resource.buttonTooltip);
            toolTip1.SetToolTip(btnBrowse, Resource.fileTooltip);
            toolTip1.SetToolTip(sourceFileName, Resource.fileTooltip);
            toolTip1.SetToolTip(btnCalcStats, Resource.statsTooltip);
            linkLabel1.Links[0].LinkData = "http://www.imapsmanager.com";
            outputFormatCb.SelectedIndex = 2;
            
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            CancelConvert = true;
        }

        private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            System.Diagnostics.Process.Start((string)e.Link.LinkData);
        }

        private void btnCompression_Click(object sender, EventArgs e)
        {
            Formats dlg = new Formats();
            dlg.mapLayer = mapLayer;
            dlg.satelliteLayer = satelliteLayer;
            dlg.hybridLayer = hybridLayer;
            dlg.mapJpgQuality = mapJpgQuality;
            dlg.satelliteJpgQuality = satelliteJpgQuality;
            dlg.hybridJpgQuality = hybridJpgQuality;
            if (dlg.ShowDialog() == DialogResult.Cancel)
                return;
            mapLayer = dlg.mapLayer;
            satelliteLayer = dlg.satelliteLayer;
            hybridLayer = dlg.hybridLayer;
            mapJpgQuality = dlg.mapJpgQuality;
            satelliteJpgQuality = dlg.satelliteJpgQuality;
            hybridJpgQuality = dlg.hybridJpgQuality;
        }

        private void btnCalcStats_Click(object sender, EventArgs e)
        {
            FileInfoDlg dlg = new FileInfoDlg();
            dlg.FileName = sourceFileName.Text;
            dlg.ShowDialog();
        }




        private void textBox1_TextChanged(object sender, EventArgs e)
        {
            if (sourceFileName.Text != "")
                btnCalcStats.Enabled = true;
            else
                btnCalcStats.Enabled = false;
        }

        private void outputFormatCb_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (OuputFormat)
            { 
                case mapFrmt.v10:
                case mapFrmt.v10hiRes:
                case mapFrmt.v10small:
                case mapFrmt.v11:
                case mapFrmt.v11hiRes:
                case mapFrmt.v11small:
                case mapFrmt.v4:
                case mapFrmt.v5:
                    btnCompression.Enabled = true;
                    break;
                default:
                    btnCompression.Enabled = false;
                    break;
            }
        }
    }
    class TileData
    {
        public Bitmap image;
        public int addedTiles = 0;
        public Graphics g;
        public ImgFormat format;
        public TileData()
            : this(128)
        { }
        public TileData(int size)
        {
            image = new Bitmap(size, size);
            g = Graphics.FromImage(image);
        }
    }
}
