unit LibJpegTran;

{* transupp.h
 *
 * Copyright (C) 1997-2012, Thomas G. Lane, Guido Vollbeding.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains declarations for image transformation routines and
 * other utility code used by the jpegtran sample application.  These are
 * NOT part of the core JPEG library.  But we keep these routines separate
 * from jpegtran.c to ease the task of maintaining jpegtran-like programs
 * that have other user interfaces.
 *
 * NOTE: all the routines declared here have very specific requirements
 * about when they are to be executed during the reading and writing of the
 * source and destination files.  See the comments in transupp.c, or see
 * jpegtran.c for an example of correct usage.

 * Although rotating and flipping data expressed as DCT coefficients is not
 * hard, there is an asymmetry in the JPEG format specification for images
 * whose dimensions aren't multiples of the iMCU size.  The right and bottom
 * image edges are padded out to the next iMCU boundary with junk data; but
 * no padding is possible at the top and left edges.  If we were to flip
 * the whole image including the pad data, then pad garbage would become
 * visible at the top and/or left, and real pixels would disappear into the
 * pad margins --- perhaps permanently, since encoders & decoders may not
 * bother to preserve DCT blocks that appear to be completely outside the
 * nominal image area.  So, we have to exclude any partial iMCUs from the
 * basic transformation.
 *
 * Transpose is the only transformation that can handle partial iMCUs at the
 * right and bottom edges completely cleanly.  flip_h can flip partial iMCUs
 * at the bottom, but leaves any partial iMCUs at the right edge untouched.
 * Similarly flip_v leaves any partial iMCUs at the bottom edge untouched.
 * The other transforms are defined as combinations of these basic transforms
 * and process edge blocks in a way that preserves the equivalence.
 *
 * The "trim" option causes untransformable partial iMCUs to be dropped;
 * this is not strictly lossless, but it usually gives the best-looking
 * result for odd-size images.  Note that when this option is active,
 * the expected mathematical equivalences between the transforms may not hold.
 * (For example, -rot 270 -trim trims only the bottom edge, but -rot 90 -trim
 * followed by -rot 180 -trim trims both edges.)
 *
 * We also offer a lossless-crop option, which discards data outside a given
 * image region but losslessly preserves what is inside.  Like the rotate and
 * flip transforms, lossless crop is restricted by the JPEG format: the upper
 * left corner of the selected region must fall on an iMCU boundary.  If this
 * does not hold for the given crop parameters, we silently move the upper left
 * corner up and/or left to make it so, simultaneously increasing the region
 * dimensions to keep the lower right crop corner unchanged.  (Thus, the
 * output image covers at least the requested region, but may cover more.)
 * The adjustment of the region dimensions may be optionally disabled.
 *
 * We also provide a lossless-resize option, which is kind of a lossless-crop
 * operation in the DCT coefficient block domain - it discards higher-order
 * coefficients and losslessly preserves lower-order coefficients of a
 * sub-block.
 *
 * Rotate/flip transform, resize, and crop can be requested together in a
 * single invocation.  The crop is applied last --- that is, the crop region
 * is specified in terms of the destination image after transform/resize.
 *
 * We also offer a "force to grayscale" option, which simply discards the
 * chrominance channels of a YCbCr image.  This is lossless in the sense that
 * the luminance channel is preserved exactly.  It's not the same kind of
 * thing as the rotate/flip transformations, but it's convenient to handle it
 * as part of this package, mainly because the transformation routines have to
 * be aware of the option to know how many components to work on. }

interface

uses
  LibJpeg8;

{$IFDEF FPC}
  {$MODE Delphi}

  {$IFDEF CPUI386}
    {$DEFINE CPU386}
    {$ASMMODE INTEL}
  {$ENDIF}

  {$IFNDEF WIN32}
    {$LINKLIB c}
  {$ENDIF}
{$ENDIF}

{$EXTENDEDSYNTAX ON}
{$ALIGN 8}
{$MINENUMSIZE 4}

type
{ Codes for supported types of image transformations. }

JXFORM_CODE = (
	JXFORM_NONE,		  { no transformation }
	JXFORM_FLIP_H,		{ horizontal flip }
	JXFORM_FLIP_V,		{ vertical flip }
	JXFORM_TRANSPOSE,	{ transpose across UL-to-LR axis }
	JXFORM_TRANSVERSE,	{ transpose across UR-to-LL axis }
	JXFORM_ROT_90,		{ 90-degree clockwise rotation }
	JXFORM_ROT_180,		{ 180-degree rotation }
	JXFORM_ROT_270,		{ 270-degree clockwise (or 90 ccw) }
	JXFORM_DROP		    { drop }
);

{ Codes for crop parameters, which can individually be unspecified,
  positive or negative for xoffset or yoffset,
  positive or forced for width or height. }
  
JCROP_CODE = (
  JCROP_UNSET,
  JCROP_POS,
  JCROP_NEG,
  JCROP_FORCE
);

{ Transform parameters struct.
  NB: application must not change any elements of this struct after
  calling jtransform_request_workspace. }
  
jpeg_transform_info_ptr = ^jpeg_transform_info;  

jpeg_transform_info = record 
  { Options: set by caller }
  transform: JXFORM_CODE;	{ image transform operator }
  perfect: JBOOL;		      { if TRUE, fail if partial MCUs are requested }
  trim: JBOOL;			      { if TRUE, trim partial MCUs as needed }
  force_grayscale: JBOOL;	{ if TRUE, convert color image to grayscale }
  crop: JBOOL;			      { if TRUE, crop source image }

  { Crop parameters: application need not set these unless crop is TRUE.
    These can be filled in by jtransform_parse_crop_spec(). }    
  crop_width: JDIMENSION;	      { Width of selected region }
  crop_width_set: JCROP_CODE;	  { (forced disables adjustment) }
  crop_height: JDIMENSION;	    { Height of selected region }
  crop_height_set: JCROP_CODE;	{ (forced disables adjustment) }
  crop_xoffset: JDIMENSION;	    { X offset of selected region }
  crop_xoffset_set: JCROP_CODE;	{ (negative measures from right edge) }
  crop_yoffset: JDIMENSION;	    { Y offset of selected region }
  crop_yoffset_set: JCROP_CODE;	{ (negative measures from bottom edge) }

  { Drop parameters: set by caller for drop request }
  drop_ptr: j_decompress_ptr;
  drop_coef_arrays: jvirt_barray_ptr_ptr;
  
  { Internal workspace: caller should not touch these }
  num_components: integer;		                  { # of components in workspace }
  workspace_coef_arrays: jvirt_barray_ptr_ptr;  { workspace for transformations }
  output_width: JDIMENSION;	                    { cropped destination dimensions }
  output_height: JDIMENSION;
  x_crop_offset: JDIMENSION;	                  { destination crop offsets measured in iMCUs }
  y_crop_offset: JDIMENSION;
  drop_width: JDIMENSION;	                      { drop dimensions measured in iMCUs }
  drop_height: JDIMENSION;
  iMCU_sample_width: integer;	                      { destination iMCU size }
  iMCU_sample_height: integer;
end;

var  
{ Parse a crop specification (written in X11 geometry style) }
jtransform_parse_crop_spec: function(info: jpeg_transform_info_ptr; spec: pAnsiChar): boolean; cdecl;

{ Request any required workspace }
jtransform_request_workspace: function(srcinfo: j_decompress_ptr; info: jpeg_transform_info_ptr): boolean; cdecl;

{ Adjust output image parameters }
jtransform_adjust_parameters: function(srcinfo: j_decompress_ptr; dstinfo: j_compress_ptr;
  src_coef_arrays: jvirt_barray_ptr_ptr; info: jpeg_transform_info_ptr): jvirt_barray_ptr_ptr; cdecl;
  
{ Execute the actual transformation, if any }
jtransform_execute_transform: procedure(srcinfo: j_decompress_ptr; dstinfo: j_compress_ptr;
  src_coef_arrays: jvirt_barray_ptr_ptr; info: jpeg_transform_info_ptr); cdecl;
  
{ Determine whether lossless transformation is perfectly
  possible for a specified image and transformation. }
jtransform_perfect_transform: function(image_width: JDIMENSION; mage_height: JDIMENSION;
  MCU_width: integer; MCU_height: integer; transform: JXFORM_CODE): boolean; cdecl;

{ jtransform_execute_transform used to be called
  jtransform_execute_transformation, but some compilers complain about
  routine names that long.  This macro is here to avoid breaking any
  old source code that uses the original name... }
  
type
{ Support for copying optional markers from source to destination file. }

JCOPY_OPTION = (
	JCOPYOPT_NONE,		  { copy no optional markers }
	JCOPYOPT_COMMENTS,	{ copy only comment (COM) markers }
	JCOPYOPT_ALL		    { copy all optional markers }
);

const
  JCOPYOPT_DEFAULT = JCOPYOPT_COMMENTS;	{ recommended default }

var
{ Setup decompression object to save desired markers in memory }
jcopy_markers_setup: procedure(srcinfo: j_decompress_ptr; option: JCOPY_OPTION); cdecl;

{ Copy markers saved in the given source object to the destination object }
jcopy_markers_execute: procedure(srcinfo: j_decompress_ptr; dstinfo: j_compress_ptr; option: JCOPY_OPTION); cdecl;

const
  LIB_JPEG_TRAN_NAME = 'transupp.dll';

function InitLibJpegTran(const LibName: AnsiString = LIB_JPEG_TRAN_NAME): boolean;
procedure QuitLibJpegTran;

implementation

uses
 SyncObjs;

var
  LibCS: TCriticalSection = nil;
  LibHandle: cardinal = 0;
  LibRefCount: Integer = 0;
  
const
  Kernel32 = 'kernel32.dll';

  function LoadLibrary(lpFileName: pAnsiChar): LongWord; stdcall; external Kernel32 name 'LoadLibraryA';
  function FreeLibrary(hModule: LongWord): LongBool; stdcall; external Kernel32 name 'FreeLibrary';
  function GetProcAddress(hModule: LongWord; lpProcName: pAnsiChar): Pointer; stdcall; external Kernel32 name 'GetProcAddress';

function GetProcAddr(Name: pAnsiChar): Pointer;
begin
  GetProcAddr := GetProcAddress(LibHandle, Name);
end;

function InitLibJpegTran(const LibName: AnsiString): boolean;
begin
  LibCS.Acquire;
  try
    if (LibRefCount = 0) or (LibHandle = 0) then begin

      if LibHandle = 0 then begin
        LibHandle := LoadLibrary(pAnsiChar(LibName));
      end;

      if LibHandle <> 0 then begin
        jtransform_parse_crop_spec := GetProcAddr('jtransform_parse_crop_spec');
        jtransform_request_workspace := GetProcAddr('jtransform_request_workspace');
        jtransform_adjust_parameters := GetProcAddr('jtransform_adjust_parameters');
        jtransform_execute_transform := GetProcAddr('jtransform_execute_transform');
        jtransform_perfect_transform := GetProcAddr('jtransform_perfect_transform');
        jcopy_markers_setup := GetProcAddr('jcopy_markers_setup');
        jcopy_markers_execute := GetProcAddr('jcopy_markers_execute');
      end;
    end;

    Result :=
      (Addr(jtransform_parse_crop_spec) <> nil) or
      (Addr(jtransform_request_workspace) <> nil) or
      (Addr(jtransform_adjust_parameters) <> nil) or
      (Addr(jtransform_execute_transform) <> nil) or
      (Addr(jtransform_perfect_transform) <> nil) or
      (Addr(jcopy_markers_setup) <> nil) or
      (Addr(jcopy_markers_execute) <> nil);

    if Result then begin
      Inc(LibRefCount);
    end;

  finally
    LibCS.Release;
  end;
end;

procedure QuitLibJpegTran;
begin
  LibCS.Acquire;
  try
    Dec(LibRefCount);

    if LibRefCount <= 0 then begin

      if LibHandle <> 0 then begin
        FreeLibrary(LibHandle);
        LibHandle := 0;
      end;

      jtransform_parse_crop_spec := nil;
      jtransform_request_workspace := nil;
      jtransform_adjust_parameters := nil;
      jtransform_execute_transform := nil;
      jtransform_perfect_transform := nil;
      jcopy_markers_setup := nil;
      jcopy_markers_execute := nil;

    end;
  finally
    LibCS.Release;
  end;
end;

initialization
  LibCS := TCriticalSection.Create;

finalization
  QuitLibJpegTran;
  LibCS.Free;

end.
