unit uSRTM;

interface

{$DEFINE TEXT_MEM_COPY}

uses dglOpenGL, GLHelper, uUtils, Geometry, ExtCtrls, glBmp, Windows, Classes, Graphics, Math, SysUtils;

const
  Height_Scale  = 0.006;

type
  TSIArray = array of array of SmallInt;
  TSRTMType = (stUnknown,st3ArcSec,st1ArcSec);
  TTransAct = (taNone, taFlipH, taFlipV, taRot180,taInvert);

  TGeoPoint = packed record
                case byte of
                  0:(Lat,Lon:Double);
                  1:(Z,X:double);
              end;

  TCoordinates = object
  private
    Origin,
    Point:TGeoPoint;
    procedure UpdatePoint;
  public
    VertexSize,RawRes:integer;
    procedure SetOrigin(Lat,Lon:double);

    procedure ParseSRTMName(Name:string);
    function GetSTRMName:string;
    procedure SetRaw(X,Z:double);
    procedure SetVertex(X,Z:double);
    procedure SetQT(X,Z:integer);
    procedure SetGeoPos(Lat,Lon:double);
    procedure SetGeoPosAbs(Lat,Lon:double);

    function GetGeoPos:TGeoPoint;
    function GetRaw:TPoint;
    function GetVertex:TGeoPoint;
  end;

  PMarkerObject = ^TMarkerObject;
  TMarkerObject = record
                    Pos:TGeoPoint;
                    Name:string[40];
                    MClass:Word;
                  end;
  PMArkerClass = ^TMarkerClass;
  TMarkerClass = record
                    ID:Word;
                    Name:string[40];
                    R,G,B:Single;
                    Visible:boolean;
                  end;

  TProgressEvent = procedure (Sender:TObject; PercentDone:byte) of object;

  TTopoView = class
  private
    FDist: single;
    FTexture: TBitmap;
    FDrawPanel: TCustomPanel;
    FHighest: integer;
    FLowest: integer;
    FVertScale: single;
    FBack: TColor;

    FTurn: single;
    FPitch: single;
    FHiking: boolean;
    FCenterY: Single;
    FCenterZ: Single;
    FCenterX: Single;
    FGradientHeight: boolean;
    FWireFrame: boolean;
    FFileName:string;
    FLightShading: boolean;
    FLoaded: boolean;
    FProgress: TProgressEvent;
    procedure SetTexture(const Value: TBitmap);
    procedure SetBackground(const Value: TColor);
    function GetHeight(const X, Z: Single): Single;
    function GetCoords: TCoordinates;
    procedure ShowText(pText: String);
    function GetTextWidth(Text: String): Double;
    procedure CreateFontLists;
  protected
    RawHeights    : TSIArray;
    FRawRes       : integer;
    VertexArray   : array of array[0..2] of Single; // Vertexdaten der Hhenkarte
    TexCoordArray : array of array[0..1] of Single; // Texturkoordinaten der Hhenkarte
    FResolution   : integer;
    DC            : HDC;
    RC            : HGLRC;
    GLTexture     : TglBMP;
    TextureGenerated : boolean;
    DL            : Cardinal;
    FSize         : integer;
    FontLists     : Cardinal;
    FABCW         : array[0..255] of TABC;
    FSeaLevel     : integer;
    Inited,
    IsSRTMData    : Boolean;


    MarkerList,
    ClassList:TList;
    function GetMarker(I:Integer):TMarkerObject;
    function FindClass(ID:Word):TMarkerClass;

    procedure DoRealDraw;   virtual;
    procedure DrawMarkers;
    procedure DoLoadReady; virtual;
    procedure DoLightMapping;
    procedure VertexPosToRawPos(var V: TVector3f);     //virtual
    procedure DoProgress(Done:Single);
  public
    constructor Create;
    destructor Destroy;  override;
    procedure InitGL;      virtual;
    procedure DrawGL;
    procedure UpdateGLSize;
    procedure ResetPos;      virtual;
    function GetDC:HDC;
    function GetRC: HGLRC;
    property Loaded : boolean read FLoaded;

    procedure LoadFromFile(FN: string; MakeText: boolean=true);
    function LoadHGT(FN: String; MakeText: boolean=true): TSRTMType;
    procedure LoadBMP(FN: string; MakeText: boolean=true);
    property CurrentFile : String read FFileName;
    procedure ResetContent;

    procedure SaveScreen(Filename:String);
    procedure SaveHeightMap(FileName:string);
    procedure SaveTexture(FileName:string);

    procedure MakeVertex(Size:integer=128);    virtual;

    property DrawPanel  : TCustomPanel read FDrawPanel  write FDRawPanel;
    property Resolution : integer      read FResolution write FResolution;
    property RawRes     : integer      read FRawRes;
    property Texture    : TBitmap      read FTexture    write SetTexture;
    procedure SetSeaLevel(Value:integer);
    procedure GenerateTexture;
    procedure UpdateTexture(Action:TTransAct=taNone);
    property VertScale  : single       read FVertScale  write FVertScale;
    property Background : TColor       read FBack       write SetBackground;
    property Lowest     : integer      read FLowest;
    property Highest    : integer      read FHighest;

    property GradientHeight:boolean  read FGradientHeight write FGradientHeight;

    property WireFrame: boolean read FWireFrame write FWireFrame;
    property LightShading: boolean read FLightShading write FLightShading;

    property Distance   : single       read FDist       write FDist;
    property Pitch      : single       read FPitch      write FPitch;
    property Turn       : single       read FTurn       write FTurn;
    property Hiking     : boolean      read FHiking     write FHiking;
    property CenterX    : Single       read FCenterX    write FCenterX;
    property CenterY    : Single       read FCenterY    write FCenterY;
    property CenterZ    : Single       read FCenterZ    write FCenterZ;

    function GetHeightMapHeight(const X,Z : Single): Single;  virtual;
    function OvergroundPos(const X, Z: Single): Single;
    property Coordinates:TCoordinates read GetCoords;
    function GetMousePos(MX, MY: integer): TCoordinates;

    procedure AddMarker(Pos:TGeoPoint; Name:String; MClass:Word);
    procedure ClearMarkers;
    property GetMarkerList : TList read MarkerList;
    procedure AddClass(ID:Word; Name:String; R,G,b:Single);
    procedure ClassVis(ID:Word; Vis:boolean);
    procedure ClearClasses;
    property GetClassList : TList read ClassList;

    property OnProgress:TProgressEvent read FProgress write FProgress;
  end;

  TVBOVertex = record
    S,T,X,Y,Z : TGLFloat;
   end;



  TTopoViewVBO = class(TTopoView)
  protected
    UseVBuffer : boolean;
    VBOSize    : Integer;
    VBOPointer : Pointer;
    VBuffer    : TGLUInt;
    Tex        : TGLUInt;

    procedure DoRealDraw;    override;
  public
    procedure InitGL;        override;

    procedure MakeVertex(Size:integer=128); override;
    property VBufCount: integer read VBOSize;
  end;

  const
    QuadTreeLength :integer = 2048+1; //Gre der Quadtree Matrix (2^n +1)


  type
//    {$A-}
    TQuadTreeEntry = Byte;

{
    TQuadTreeMatrix = Array[0..QuadTreeLength-1, 0..QuadTreeLength-1] of
                        TQuadTreeEntry;

    TQuadTreeD2Matrix = Array[0..QuadTreeLength-1, 0..QuadTreeLength-1] of
                          TQuadTreeD2Entry;
}
    TQuadTreeMatrix   = Array of array of TQuadTreeEntry;

//    {$A+}

  TTopoViewCLOD = class(TTopoView)
  private
    procedure CreateQuadTree;
    procedure RenderQuadTree;
    procedure DoD2CrackAvoid;
    procedure QTPosToVertexPos(var V: TVector3f);    virtual;
    function GetHeightMapHeightQT(const X, Z: Single): Single;  virtual;
    procedure QTPosToRawPos(var V: TVector3f);                  virtual;
  protected
    QuadMatrix : TQuadTreeMatrix;
    D2Matrix : TQuadTreeMatrix;
    KameraPosition : TVector3f;
    QTSize:integer;
    procedure DoRealDraw;    override;
    procedure ZeroQT(var qt:TQuadTreeMatrix);
  public
    C : Single;         //Groes C
    kleinC : Single;    //Kleines c
    procedure InitGL;       override;
    procedure MakeVertex(Size: integer=128);  override;
  end;




var  wglSwapIntervalEXT    : function(interval: TGLint) : BOOL;               stdcall = nil;


function FormatGeoPoint(Point:TGeoPoint):string;
function GeoPoint(Lon,Lat:double):TGeoPoint;

implementation
{
// Not used right now
resourcestring
   sLoading    = 'Loading Data...';
   sBuildWorld = 'Building World...';
   sCreatingNormals = 'Creating Normals...';
   sCreatingLightMap = 'Creating Lightmap...';
   sSavingHM         = 'Saving Heightmap...';

}


const sTempText = 'text.bmp';
      i3ArcSize = 1201 * 1201 * 2;     // 1442401  points =  2884802 bytes
      i1ArcSize = 3601 * 3601 * 2;     //12967201  points = 25934402 bytes

      iNoData   = -32768;

      cX        = 0;
      cY        = 1;
      cZ        = 2;
      cH        = cY;
      cW        = 3;

function FormatGeoPoint(Point:TGeoPoint):string;
var NS,WE:Char;
begin
  if Point.Lat<0 then NS:= 'S' else NS:='N';
  if Point.Lon<0 then WE:= 'W' else WE:='E';

  Result:= format('%3.6f%s  %3.6f%s',[abs(Point.Lon),WE,abs(Point.lat),NS]);
end;

function GeoPoint(Lon,Lat:double):TGeoPoint;
begin
  Result.Lat:= Lat;
  Result.Lon:= Lon;
end;

function Fetch(var s:string; num:integer):string;
begin
  Result:= Copy(s,1,num);
  delete(s,1,num);
end;

{ TCoordiantes }

procedure TCoordinates.ParseSRTMName(Name: string);
var s:string;
    la,lo:double;
begin
  if IsFormat(ExtractFileName(Name),'C00C000.*') then begin
    s:= ChangeFileExt(ExtractFileName(Name),'');
    if fetch(s,1)='N' then la:= 1 else la:= -1;
    la:= la * strtoint(fetch(s,2));
    if fetch(s,1)='E' then lo:= 1 else lo:= -1;
    lo:= lo * strtoint(fetch(s,3));
    SetOrigin(la,lo);
  end else SetOrigin(0,0);
end;

function TCoordinates.GetSTRMName: string;
var ns,we:char;
    P:TGeoPoint;
begin
  p:= GetGeoPos;
  if p.Lat>=0 then ns:= 'N' else ns:= 'S';
  if p.Lon>=0 then we:= 'E' else we:= 'W';
  Result:= format('%s%.2d%s%.3d',[ns,round(abs(p.Lat)),we,round(abs(p.Lon))]);
end;


procedure TCoordinates.SetOrigin(Lat, Lon: double);
begin
  Origin.Lat:= Lat;
  Origin.Lon:= Lon;
end;

procedure TCoordinates.SetRaw(X, Z: double);
begin
  Point.X:= X/RawRes;
  Point.Z:= Z/RawRes;
  UpdatePoint;
end;

procedure TCoordinates.SetVertex(X, Z: double);
begin
  Point.X:= ((X+VertexSize/2) / VertexSize);
  Point.Z:= ((Z+VertexSize/2) / VertexSize);
  UpdatePoint;
end;

procedure TCoordinates.SetQT(X, Z: integer);
begin
  Point.X:= X / (QuadTreeLength - 1);
  Point.Z:= Z / (QuadTreeLength - 1);
  UpdatePoint;
end;

procedure TCoordinates.SetGeoPos(Lat, Lon: double);
begin
  Point.Lat:= Lat;
  Point.Lon:= Lon;
end;

procedure TCoordinates.SetGeoPosAbs(Lat, Lon: double);
begin
  Point.Lat:= Lat-Origin.Lat;
  Point.Lon:= Lon-Origin.Lon;
end;

procedure TCoordinates.UpdatePoint;
begin
  Point.Lon:= 1-Point.Lon;
end;

function TCoordinates.GetGeoPos: TGeoPoint;
begin
  Result.Lat:= Origin.Lat+Point.Lat;
  Result.Lon:= Origin.Lon+Point.Lon;
end;

function TCoordinates.GetRaw: TPoint;
begin
  Result.X:= trunc(Point.Lat*RawRes);
  Result.Y:= trunc(Point.Lat*RawRes);
end;

function TCoordinates.GetVertex: TGeoPoint;
begin
  Result.X:= (Point.X*VertexSize)-(VertexSize/2);
  Result.Z:= (VertexSize/2)-(Point.Z*VertexSize);
end;



type TColorStage = record H:Smallint; C:TColor end;
     TRGB = Array[0..2] of Byte;


var Colors: array of TColorStage;
    DoGradientHeight:boolean = false;

procedure AddHeight(H:smallint; color:TColor);
begin
  SetLength(Colors,length(Colors)+1);
  Colors[high(Colors)].H:= H;
  Colors[high(Colors)].C:= color;
end;

procedure MakeRGB(Color : TColor; var RGB : TRGB);
begin
  Move(Color, RGB[0], 3)
end;

procedure MakeColor(RGB : TRGB; var Color : TColor);
begin
  Move(RGB[0], Color,3)
end;

function MakeHeightColor(H:Smallint; SeaLevel:integer=0):TColor;
var hg:Smallint;
    i:integer;
    upper,lower:TColorStage;
    LevelDifference : LongInt; //Levelunterschied zw. Upper u. Lower
    LowerDifference : LongInt; //Abstand HeightmapHeight - Lower
    LowerRGB, UpperRGB : TRGB; //RGB werte von Upper Lower
    Prozent : Single;          //Prozent der eingesetzten Farbwerte
    FinalRGB : TRGB;           //End RGB Wert
begin
  if not(H=iNoData) then
    hg:= H-SeaLevel
    else begin
      Result:= colors[0].C;
      exit;
    end;
  for i:= 0 to High(Colors) do begin
    if hg<=Colors[i].H then begin
      upper:= Colors[i];
      if i>0 then lower:= Colors[i-1] else lower:= upper;
      break;
    end;
  end;
  LevelDifference := Upper.H - lower.H;
  if (LevelDifference>0) and DoGradientHeight then begin
    LowerDifference := hg-lower.H;
    Prozent := LowerDifference / LevelDifference;

    MakeRGB(lower.C,LowerRGB);
    MakeRGB(upper.C,UpperRGB);

    FinalRGB[0] := Trunc((UpperRGB[0] - LowerRGB[0])*Prozent + LowerRGB[0]);
    FinalRGB[1] := Trunc((UpperRGB[1] - LowerRGB[1])*Prozent + LowerRGB[1]);
    FinalRGB[2] := Trunc((UpperRGB[2] - LowerRGB[2])*Prozent + LowerRGB[2]);
      //    y      =       (            m            )*   x    +      t
    Result:= RGB(FinalRGB[0],FinalRGB[1],FinalRGB[2]);
  end else Result:= upper.C;
end;

procedure InitHeights;
begin
  SetLength(colors,0);
  AddHeight(low(SmallInt), clBlue);
{  AddHeight(0    , $00FF8000);
  AddHeight(7    , $00D9FE81);
  AddHeight(20   , clGreen);
  AddHeight(50   , $00009D4F);
  AddHeight(150  , $0000D26F);
  AddHeight(300  , $0000C6B7);
  AddHeight(500  , $0070FEE1);
  AddHeight(800  , $004AD7FF);
  AddHeight(1000 , $000099FF);
  AddHeight(1500 , $000080FF);
  AddHeight(2000 , $002666BB);
  AddHeight(2800 , $001743A8);
  AddHeight(3500 , $00253A8D);
  AddHeight(5000 , $001616B4);
  AddHeight(6000 , $00C1C1FF);         }
  AddHeight(0    , rgb(  8,  0,132));
  AddHeight(10   , rgb(  0, 99,  0));
  AddHeight(30   , rgb( 57,148, 33));
  AddHeight(100  , rgb(181,222,132));
  AddHeight(150  , rgb(247,247,156));
  AddHeight(200  , rgb(247,247,132));
  AddHeight(250  , rgb(239,222, 57));
  AddHeight(300  , rgb(214,181,  8));
  AddHeight(350  , rgb(189,140, 24));
  AddHeight(450  , rgb(173,115, 41));
  AddHeight(1500 , rgb(132, 49, 24));
  AddHeight(2000 , rgb(107,  8,  0));
  AddHeight(3000 , rgb( 99,  0,  0));
  AddHeight(4200 , rgb(155,  0,  0));
  AddHeight(5000 , rgb(216, 77, 61));
  AddHeight(6500 , rgb(227,132,121));
  AddHeight(7250 , rgb(236,171,164));
  AddHeight(8000 , rgb(242,197,191));
  AddHeight(high(SmallInt), $00FFFFFF);
end;

procedure BlendBmp(Target,Src,Alpha:TBitmap);
var i,j:integer;
    a:single;
    sr,al,ta:PByteArray;
begin
  for i:= 0 to Src.Height-1 do begin
    sr:= Src.ScanLine[i];
    al:= Alpha.ScanLine[i];
    ta:= Target.ScanLine[i];
    j:= 0;
    while j< Src.Width*4 do begin
      a:= al^[j]/255;
      ta^[j]  := trunc(sr^[j]*a);        // blue
      ta^[j+1]:= trunc(sr^[j+1]*a);      // green
      ta^[j+2]:= trunc(sr^[j+2]*a);      // red
      inc(j,4);
    end;
  end;
end;

procedure ClearList(L:TList);
begin
  while l.Count >0 do begin
    Dispose(l[0]);
    l.Delete(0);
  end;
end;



{ TTopoView }

constructor TTopoView.Create;
begin
  inherited;
  Inited:= false;
  IsSRTMData:= false;
  GLTexture := TglBMP.Create;
  FTexture:= TBitmap.Create;
  ResetPos;
  FResolution:= 256;
  FWireFrame:= false;
  FLightShading:= true;
  MarkerList:= TList.Create;
  ClassList:= TList.Create;
  InitHeights;
  ResetContent;
end;

destructor TTopoView.Destroy;
begin
  if Inited then begin
    DeactivateRenderingContext;
    DestroyRenderingContext(RC);
  end;
  FTexture.Free;
  GLTexture.Free;
  ClearMarkers;   MarkerList.Free;
  ClearClasses;    ClassList.Free;
  inherited;
end;

procedure TTopoView.VertexPosToRawPos(var V: TVector3f);
begin
  V[cX] := ((V[cX]+FSize/2) / FSize) * FRawRes;
  V[cZ] := ((V[cZ]+FSize/2) / FSize) * FRawRes;
end;


function TTopoView.GetHeight(const X,Z : Single): Single;
begin
  if (X>=0) and (Z>=0) and
     (X<=FRawRes) and (Z<=FRawRes) then
      Result := RawHeights[trunc(X),trunc(Z)]
  else
    Result := iNoData;
  if Result=iNoData then
    Result:= FLowest-1;
  Result:= Result*Height_Scale;
end;


function TTopoView.GetHeightMapHeight(const X,Z : Single): Single;
////////////////////////////////////////////////////////////////////////////////
// GetHeightMapHeight
//------------------------------------------------------------------------------
// Holt die Hhe der Map an der Stelle X,Z (Als Vertex-Position)
//------------------------------------------------------------------------------
// Parameter: X,Z: Koordinaten
// Rckgabe : Hhe der Karte an X,Z
////////////////////////////////////////////////////////////////////////////////
var
  v: TVector3f;
begin
  v[cX]:= X;
  v[cZ]:= Z;
  VertexPosToRawPos(v);
  Result:= GetHeight(v[cx],v[cZ]);
end;

function TTopoView.OvergroundPos(const X,Z : Single): Single;
begin
  Result:= GetHeightMapHeight(X,Z)+(5*Height_Scale);
end;

function TTopoView.GetDC: HDC;
begin
  Result:= DC;
end;

function TTopoView.GetRC: HGLRC;
begin
  Result:= RC;
end;

procedure TTopoView.InitGL;
var
  c:TRGBA;
begin
  InitOpenGL;
  DC := Windows.GetDC(FDrawPanel.Handle);
  RC := CreateRenderingContext(DC, [opDoubleBuffered], 32, 24, 0, 0, 0, 0);
  ActivateRenderingContext(DC, RC);
  c:= ColorToRGBA(FBack);
  glClearColor(c.R,c.G,c.B,c.A);
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_DEPTH_TEST);
  wglSwapIntervalEXT := wglGetProcAddress('wglSwapIntervalEXT');
  wglSwapIntervalEXT(0);
  CreateFontLists;
  Inited:= true;
end;



procedure TTopoView.UpdateGLSize;
begin
  glMatrixMode(GL_PROJECTION);
  glViewport(0, 0, FDrawPanel.ClientWidth, FDrawPanel.ClientHeight);
  glLoadIdentity;
  gluPerspective(FDist, FDrawPanel.ClientWidth/FDrawPanel.ClientHeight, 0.1, FSize*4);
end;

procedure TTopoView.DrawGL;
var m:TMatrix;
begin
  UpdateGLSize;
  glMatrixMode(GL_MODELVIEW);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);
  if FWireFrame then
    GlPolygonMode(GL_FRONT, GL_LINE)
  else
    GlPolygonMode(GL_FRONT, GL_FILL);

  if not Loaded then exit;

  m:= IdentityHmgMatrix;
  m:= Geometry.Pitch(m, DegToRad(Pitch));
  m:= Geometry.Turn(m, DegToRad(Turn));

  glLoadIdentity;

  if not FHiking then
    glTranslatef(0,0,-FSize*2);
  glMultMatrixf(@m[0,0]);
  if FHiking then
    glTranslatef(FCenterX, -FCenterY, FCenterZ)
  else
    glTranslatef(FCenterX, 0, FCenterZ);

  glScalef(1, FVertScale, 1);

  glColor3f(1.0,1.0,1.0);

  GLTexture.Bind;
  DoRealDraw;
  
  glBindTexture(GL_TEXTURE_2D, 0);
  if IsSRTMData then
    DrawMarkers;

  SwapBuffers(DC);
end;

procedure TTopoView.DoRealDraw;
begin
  glVertexPointer(3, GL_FLOAT, 12, VertexArray);
  glTexCoordPointer(2, GL_FLOAT, 8, TexCoordArray);
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  if Length(VertexArray) > 0 then
    glDrawArrays(GL_QUADS, 0, Length(VertexArray));
end;


procedure TTopoView.LoadBMP(FN: string; MakeText: boolean);
var BMP:TBitmap;
    X,Z:integer;
    pc:TColor;
begin
  SetLength(VertexArray,0);
  SetLength(TexCoordArray,0);
  DoProgress(0);
  BMP := TBitMap.Create;
  try
    BMP.LoadFromFile(FN);
    FRawRes:= BMP.Width-1;
    SetLength(RawHeights, FRawRes+1, FRawRes+1);
    for x := 0 to FRawRes do begin
      for z := 0 to FRawRes do begin
        pc:= Bmp.Canvas.Pixels[X,Z];
        RawHeights[x,z] := Trunc(pc/clWhite * 255)*4;
      end;
      DoProgress(X/FRawRes);
    end;
  finally
    Bmp.Free;
  end;
  if MakeText then GenerateTexture;
  DoLoadReady;
end;

function TTopoView.LoadHGT(FN: String; MakeText: boolean=true): TSRTMType;

  procedure ReadRawFile;
{  type TSIRec = packed record
                  case byte of
                    0: (LSB,MSB:byte);
                    1: (Sum:SmallInt);
                end;  }
  var x,z:integer;
//      temp: TSIRec;
      tmp2: smallint;
      hm  : TMemoryStream;
  begin
    hm:= TMemoryStream.Create;
    try
      hm.LoadFromFile(FN);
      if hm.Size=i3ArcSize then Result:= st3ArcSec;
      if hm.Size=i1ArcSize then Result:= st1ArcSec;
      case Result of
        st3ArcSec: FRawres:= 1200;
        st1ArcSec: FRawres:= 3600;
        stUnknown: raise Exception.Create('Wrong Size of hgt File');
      end;

      FLowest:= 9999999;
      FHighest:= -9999999;
      SetLength(RawHeights, FRawres+1, FRawres+1);
      DoProgress(0);
      for x := 0 to FRawres do begin
        for z := 0 to FRawres do begin
{

          temp.Sum:= 0;
          hm.Read(temp.MSB,1);
          hm.Read(temp.LSB,1);
          tmp2:= temp.Sum;
          RawHeights[z,x]:= tmp2;
}

          hm.Read(tmp2,2);
          RawHeights[z,x]:= ((tmp2 and $FF) shl 8) or ((tmp2 and $FF00) shr 8);
          tmp2:= RawHeights[z,x];
          if (tmp2<>iNoData) then begin
            if tmp2<FLowest then FLowest:= tmp2;
            if tmp2>FHighest then FHighest:= tmp2;
          end;
        end;
        DoProgress(X/FRawRes);
      end;
    finally
      hm.Free;
    end;
  end;

begin
  Result:= stUnknown;
  SetLength(VertexArray,0);
  SetLength(TexCoordArray,0);
  ReadRawFile;
  if MakeText then GenerateTexture;
  DoLoadReady;
end;

procedure TTopoView.DoLoadReady;
begin
  FLoaded:= true;
end;

procedure TTopoView.LoadFromFile(FN: string; MakeText: boolean);
begin
  FFileName:= FN;
  if (CompareText(ExtractFileExt(fn), '.hgt') = 0) then
    LoadHGT(fn,MakeText) else
  if (CompareText(ExtractFileExt(fn), '.bmp') = 0) then
    LoadBMP(fn,MakeText) else
  ResetContent;
  IsSRTMData:= CompareText(ExtractFileExt(FFileName),'.hgt')=0;
end;

procedure TTopoView.MakeVertex(Size:integer=128);
var x,z,i,a:integer;
   Vertex     : array[0..3] of record X,Y,Z:single; end;
   rs:single;
const
  VertPos : array[0..3] of array[0..1] of Byte = ((0,0),(0,1),(1,1),(1,0));
begin
  // Vertexe und Texturcoordinaten generien
  SetLength(TexCoordArray, (FResolution-1)*(FResolution-1)*4);
  SetLength(VertexArray, (FResolution-1)*(FResolution-1)*4);
  a:= 0;

  FSize:= Size;
  rs:= FRawres/FResolution;
  DoProgress(0);
  for x := 0 to FResolution-2 do begin
    for z := 0 to FResolution-2 do begin
      for i := 0 to 3 do begin
        Vertex[i].X := (X + VertPos[I][0])/FResolution*Size-Size/2;
        Vertex[i].Z := (Z + VertPos[I][1])/FResolution*Size-Size/2;
        Vertex[i].Y := GetHeight(trunc((X + VertPos[I][0])*rs),trunc((Z + VertPos[I][1])*rs));

        if Vertex[i].Y=iNoData then
          Vertex[i].Y:= FLowest-1;

        TexCoordArray[a][0] := (Vertex[i].X-Size/2)/Size;
        TexCoordArray[a][1] := (-Vertex[i].Z-Size/2)/Size;
        VertexArray[a][cX] := Vertex[i].X;
        VertexArray[a][cH] := Vertex[i].Y;
        VertexArray[a][cZ] := Vertex[i].Z;
        inc(a);
      end;
    end;
    DoProgress(X/(FResolution-2));
  end;
end;

procedure TTopoView.SetBackground(const Value: TColor);
begin
  FBack := Value;
end;

procedure TTopoView.UpdateTexture(Action:TTransAct=taNone);
{$IFNDEF TEXT_MEM_COPY}
var f:string;
begin
  f:= GetTempDir+sTempText;
  DeleteFile(f);
  FTexture.SaveToFile(f);
  GLTexture.LoadImage(f);
{$ELSE}
begin
  CopyBitmap(FTexture,GLTexture);
{$ENDIF}
  case Action of
    taFlipV : GLTexture.FlipVert;
    taFlipH : GLTexture.FlipHorz;
    taRot180: GLTexture.Rotate180;
    taInvert: GLTexture.Invert;
  end;
  GLTexture.GenTexture(true,false);
end;

procedure TTopoView.SetTexture(const Value: TBitmap);
begin
  FTexture.Assign(Value);
  UpdateTexture;
end;

procedure TTopoView.GenerateTexture;
begin
  DoGradientHeight:= FGradientHeight;

  DoLightMapping;
  TextureGenerated:= true;     
end;

procedure TTopoView.SetSeaLevel(Value: integer);
begin
  FSeaLevel:= Value;
  if TextureGenerated then    // only if we made the texture
    GenerateTexture;
  UpdateTexture;
end;

procedure TTopoView.DoLightMapping;
type
  TNormalMap = Array of Array[0..2] of ShortInt;

var
  NormalMap : TNormalMap;

  procedure CreateNormalMap;
  const
    VertexPoss : Array[0..8] of Array[0..1] of ShortInt =
      ((0,0),(1,-1),(1,0),(1,1),(0,1),(-1,1),(-1,0),(-1,-1),(0,-1));
    Faces : Array[0..7] of Array[0..2] of Byte =
      ((0,1,8),(0,2,1),(0,4,2),(4,3,2),(0,5,4),(0,6,5),(0,8,6),(8,7,6));
  var
    X,Z : LongInt;
    I : LongInt;

    Vertieces : Array[0..8] of TVertex;
    Normals   : Array[0..7] of TVertex;
    VertexPosMultiply : LongInt;
  begin
    SetLength(NormalMap, FResolution*FResolution);
    FillChar(Vertieces[0], SizeOf(Vertieces), 0);
    DoProgress(0);
    for Z := 0 to FResolution - 1 do begin
      for X := 0 to FResolution - 1 do
      begin
        VertexPosMultiply:= 1;
        //Vertieces holen
        for I := 0 to 8 do
        begin
          Vertieces[I][cX] := (X + VertexPoss[I][0]*VertexPosMultiply)*FRawres/FResolution;
          Vertieces[I][cZ] := (Z + VertexPoss[I][1]*VertexPosMultiply)*FRawres/FResolution;
          if (Z + VertexPoss[I][1]*VertexPosMultiply>0) and
             (X + VertexPoss[I][0]*VertexPosMultiply>0) and
             (Z + VertexPoss[I][1]*VertexPosMultiply<FResolution) and
             (X + VertexPoss[I][0]*VertexPosMultiply<FResolution) then
          Vertieces[I][cH] := GetHeight(trunc(Vertieces[I][cX]), trunc(Vertieces[I][cZ]));
        end;
        //Normalen holen
        for I := 0 to 7 do
        begin
          Normals[I] := VectorCrossProduct(
            VectorNormalize(VectorSubtract(Vertieces[Faces[I][cX]],
               Vertieces[Faces[I][cH]])),
            VectorNormalize(VectorSubtract(Vertieces[Faces[I][cH]],
               Vertieces[Faces[I][cZ]])));
          Normals[I] := VectorNormalize(Normals[I])
        end;
        //Den "Durchschnittsvektor"...
        for I := 1 to 7 do
          Normals[0] := VectorAdd(Normals[0], Normals[I]);
        Normals[0] := VectorScale(VectorNormalize(Normals[0]),127.0);
        NormalMap[Z*(FResolution) + X][cX] := Trunc(Normals[0][cX]);
        NormalMap[Z*(FResolution) + X][cH] := Trunc(Normals[0][cH]);
        NormalMap[Z*(FResolution) + X][cZ] := Trunc(Normals[0][cZ])
      end;
      DoProgress(Z/(FResolution-1));
    end;
  end;

  procedure CreateLightMap(Sun : TVertex; LightMap : TBitMap;
    LightAmbient, LightDiffuse : Single; Raytracing : Boolean);

  type
    TRay = record
             Origin, Direction : TVertex
           end;

    function RayIntersected(Ray : TRay):Boolean;
    var
       Position : TVertex;
       Vor : TVertex;
    begin
      Result := False;
      Position := Ray.Origin;
      Vor := VectorNormalize(Ray.Direction);
      Position := VectorAdd(Position, Vor);
      while (Position[0] > 0) and (Position[0] < FResolution - 1) and
        (Position[2] > 0) and (Position[2] < FResolution - 1) do
      begin
        if GetHeight(Position[0]*FRawRes/FResolution,
                     Position[2]*FRawRes/FResolution) >= Position[1] then
        begin
          Result := True;
          Break
        end;
        Position := VectorAdd(Position, Vor)
      end
    end;

  var
    X,Z : LongInt;
    Ray : TRay;
    n : TVertex;
    f : Single;
    ax,az: LongInt;
    t    : SmallInt;
    Line : PByteArray;
    c    : TColor;
    rgb  : TRGB;
  begin
    //Wir haben paraleles Licht, also bleibt der Richtungsvektor des
    //Rays immer gleich
    Ray.Direction := VectorNormalize(Sun);
    LightMap.Width:= FResolution;
    LightMap.Height:= FResolution;
    LightMap.PixelFormat:= pf32bit;

    DoProgress(0);
    for Z := 0 to FResolution - 1 do begin
      Line:= FTexture.ScanLine[z];
      for X := 0 to FResolution - 1 do begin
        ax:= trunc(X*FRawres/FResolution);
        az:= trunc(Z*FRawres/FResolution);
        t:= RawHeights[ax,az];
        c:= MakeHeightColor(t, FSeaLevel);
        if FLightShading then begin
          Ray.Origin[cX] := X;
          Ray.Origin[cZ] := Z;
          Ray.Origin[cH] := t;
          if RayTracing and RayIntersected(Ray) then
            f := LightAmbient
          else begin
            //Komprimierte Normale zurckverwandeln
            n[0] := NormalMap[Z*(FResolution) + X][cX];
            n[1] := NormalMap[Z*(FResolution) + X][cH];
            n[2] := NormalMap[Z*(FResolution) + X][cZ];

            f := LightAmbient+LightDiffuse*(VectorDotProduct(
                 VectorNormalize(n),VectorNormalize(Ray.Direction)));
            //Zuviel licht?
            if f > 1.0 then f := 1.0;
            if f < 0.0 then f := 0.0
          end
        end else f:= 1;
        MakeRGB(c,rgb);
        Line[x*4]  := Trunc(rgb[2]*f);
        Line[x*4+1]:= Trunc(rgb[1]*f);
        Line[x*4+2]:= Trunc(rgb[0]*f);
      end;
      DoProgress(Z/(FResolution-1));
    end;
  end;

var
  Sun : TVertex;

const
(*
In OpenGl gibt es unterschiedliche Arten des Lichts:

Umgebungs Licht (Ambient) ist eine Art des Lichts, dessen Richtung nicht zu erkennen ist.
  Es ensteht durch mehrfache Refelktion an Wnden oder anderen Flchen. Entsprechend gut
  vertreten ist es in Rumen, aber auch in der freien Natur umgibt uns nicht zu wenig davon.
"Diffuses" Licht(Diffuse) dagegen lsst die Richtung aus der es kommt erkennen.
  Je strker eine Seite in das Licht gehalten wird, desto heller erscheint die Oberflche.
  Je weiter man sie herausdreht, desto dunkler wird die Oberflche. Die eintreffenden
  Lichtstrahlen werden in alle Richtungen verteilt, so dass die Oberflche von allen
  Augenpositionen gleich hell erscheint.
*)

  AmbientLight =   0.60;
  DiffuseLight =   0.80;
  SunElev      =   0.50;
  SunAzi       = 225.00;    //North/East
  RayTracing   =  false;
begin
  Sun[cX] := Sin(SunAzi *Pi/180);
  Sun[cY] := SunElev;
  Sun[cZ] := Cos(SunAzi *Pi/180);

  CreateNormalMap;

  DoGradientHeight:= FGradientHeight;
  try
    CreateLightMap(Sun, Texture,
                   AmbientLight, DiffuseLight, RayTracing);
  finally
  end;
end;

procedure TTopoView.ResetContent;
begin
  TextureGenerated:= false;
  FLoaded:= false;
  GLTexture.InitEmpty(0,0);
  SetLength(RawHeights,0,0);
end;

procedure TTopoView.ResetPos;
begin
  FTurn:= -35;
  if FHiking then
    FPitch:= 20 else
    FPitch:= -35;
  FDist:= 32;
  FVertScale:= 1.0;
  FCenterX:= 0;
  FCenterY:= 0;
  FCenterZ:= 0;
  FSeaLevel:= 0;
end;

function TTopoView.GetCoords: TCoordinates;
begin
  Result.ParseSRTMName(FFileName);
  Result.VertexSize:= FSize;
  Result.RawRes:= FRawRes;
  Result.SetVertex(FCenterX,FCenterZ)
end;

function TTopoView.GetMousePos(MX,MY:integer): TCoordinates;
var modelview ,
    projection: TGLMatrixd4;
    viewport  : TGLVectori4;
    Z         : GLfloat;
    Y_new     : GLint;
    Res       : TGLVectord3;
begin
  Result:= Coordinates;

  glGetDoublev(GL_MODELVIEW_MATRIX, @modelview ); //Aktuelle Modelview Matrix in einer Variable ablegen
  glGetDoublev(GL_PROJECTION_MATRIX, @projection ); //Aktuelle Projection[s] Matrix in einer Variable ablegen
  glGetIntegerv(GL_VIEWPORT, @viewport ); // Aktuellen Viewport in einer Variable ablegen
  Y_new := viewport[3] - MY;// In OpenGL steigt Y von unten (0) nach oben

  // Auslesen des Tiefenpuffers an der Position (X/Y_new)
  glReadPixels(MX, Y_new, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT, @Z );
  // Errechnen des Punktes welcher mit den beiden Matrizen multipliziert (X/Y_new/Z) ergibt:
  gluUnProject(MX, Y_new, Z,modelview, projection, viewport,@Res[0], @Res[1], @Res[2]);

  res[cX]:= -res[cX];
  res[cZ]:= -res[cZ];
  limit(res[cX],Result.VertexSize/2);
  limit(res[cZ],Result.VertexSize/2);
  Result.SetVertex(res[cX],res[cZ]);
end;




procedure TTopoView.CreateFontLists;// Wird einmal am Anfang aufgerufen
var
  CustomFont:HFont;
begin
  FontLists := glGenLists(256);
  (*CustomFont := GetStockObject (SYSTEM_FONT);*)//Eine Alternative zu CreateFont
  CustomFont := CreateFont(
                           22,                  // Hhe
                           0,                   // Breite 0=Keine Vorgabe
                           0,
                           0,
                           0,                   // Fett?
                           0,                   // Kursiv?
                           0,                   // Unterstrichen?
                           0,                   // Durchgestrichen?
                           ANSI_CHARSET,
                           OUT_TT_PRECIS,
                           CLIP_DEFAULT_PRECIS,
                           NONANTIALIASED_QUALITY,
                           FF_DONTCARE or DEFAULT_PITCH,
                           'Arial'); // Name der Schrift
  SelectObject(DC, CustomFont);
  wglUseFontBitmaps (DC, 0, 255, FontLists);
  GetCharABCWidths(DC, 0, 255, FABCW[0]);
end;

function TTopoView.GetTextWidth(Text: String): Double;
var
  Idx: Integer;
begin
  Result := 0;
  for Idx := 1 to Length(Text) do
    Result := Result + FAbcW[Ord(Text[Idx])].abcB;
end;

procedure TTopoView.ShowText(pText:String);
begin
   glListBase(FontLists); //Liste auswhlen
   glCallLists(Length(pText), GL_UNSIGNED_BYTE, Pointer(pText));//Entsprechende Listen aufrufen
end;

procedure TTopoView.DrawMarkers;
const  TS  =  0.2;          //Pyramid Size
var i:integer;
    m:TMarkerObject;
    mc:TMarkerClass;
    c:TCoordinates;
    v:TVertex;

begin
  c.ParseSRTMName(FFileName);
  c.VertexSize:= FSize;
  c.RawRes:= FRawRes;
  glDisable(GL_LIGHTING);
  for i:= 0 to MarkerList.Count-1 do begin
    m:= GetMarker(i);
    mc:= FindClass(m.MClass);
    if mc.Visible then begin
      c.SetGeoPosAbs(m.Pos.lat,m.Pos.Lon);
      v[cX]:= c.GetVertex.X;
      v[cZ]:= c.GetVertex.Z;
      v[cY]:= OvergroundPos(v[cX],V[cZ]);
      if (v[cX]>=-(FSize/2)) and (v[cX]<=(FSize/2)) and
         (v[cZ]>=-(FSize/2)) and (v[cZ]<=(FSize/2)) then begin
        glColor3f(mc.r,mc.g,mc.B);
        glBegin(GL_TRIANGLE_FAN);
          glVertex3fv(@V[0]);
          glVertex3f(V[cX]-TS,V[cY]+4*TS,V[cZ]-TS);
          glVertex3f(V[cX]-TS,V[cY]+4*TS,V[cZ]+TS);
          glVertex3f(V[cX]+TS,V[cY]+4*TS,V[cZ]+TS);
          glVertex3f(V[cX]+TS,V[cY]+4*TS,V[cZ]-TS);
          glVertex3f(V[cX]-TS,V[cY]+4*TS,V[cZ]-TS);
        glEnd();
        glColor3f(0,0.0,0);//Aktuelle Farbe fr glRasterPos festlegen
        glRasterPos3f(V[cX],V[cY]+4.2*TS,V[cZ]); //(sichtbare) Rasterposition eintellen
        ShowText(m.Name);
      end;
    end;
  end;
end;

function TTopoView.GetMarker(I: Integer): TMarkerObject;
begin
  Result:= PMarkerObject(MarkerList[i])^;
end;

procedure TTopoView.AddMarker(Pos: TGeoPoint; Name: String; MClass:Word);
var p:PMarkerObject;
begin
  New(p);
  p^.Pos:= Pos;
  p^.Name:=Name;
  p^.MClass:= MClass;
  MarkerList.Add(p);
end;

procedure TTopoView.AddClass(ID: Word; Name:String; R, G, b: Single);
var p:PMArkerClass;
begin
  New(p);
  p^.ID:= ID;
  p^.Name:=Name;
  p^.R:= R;
  p^.G:= G;
  p^.B:= B;
  p^.Visible:= true;
  ClassList.Add(p);
end;

procedure TTopoView.ClassVis(ID: Word; Vis: boolean);
var i:integer;
    m:PMarkerClass;
begin
  for i:= 0 to ClassList.Count-1 do begin
    m:= PMarkerClass(ClassList[i]);
    if m^.id=ID then
      m^.Visible:= Vis;
  end;
end;

function TTopoView.FindClass(ID: Word): TMarkerClass;
var i:integer;
begin
  for i:= 0 to ClassList.Count-1 do begin
    Result:= PMarkerClass(ClassList[i])^;
    if Result.id=ID then
      exit;
  end;
  FillChar(result,sizeof(result),0);
end;

procedure TTopoView.SaveHeightMap(FileName: string);
var BMP:TBitmap;
    X,Z:integer;
    pc:TColor;
    Line:PByteArray;
begin
  if not FLoaded then exit;
  BMP := TBitMap.Create;
  try
    BMP.Width:= FRawRes;
    BMP.Height:= FRawRes;
    DoProgress(0);
    for z := 0 to FRawRes do begin
      Line:= Bmp.ScanLine[z];
      for x := 0 to FRawRes do begin
        pc:= trunc(RawHeights[x,z]/4);
        Line[x*4]:= pc;
        Line[x*4+1]:= pc;
        Line[x*4+2]:= pc;
      end;
      DoProgress(Z/FRawRes);
    end;
    BMP.SaveToFile(FileName);
  finally
    Bmp.Free;
  end;
end;

procedure TTopoView.SaveScreen(Filename: String);
begin
  GLTexture.SaveScreen(FileName);
end;

procedure TTopoView.SaveTexture(FileName: string);
begin
  GLTexture.SaveImage(FileName);
end;

procedure TTopoView.DoProgress(Done: Single);
begin
  if Assigned(FProgress) then
    FProgress(Self,round(Done*100));
end;

procedure TTopoView.ClearClasses;
begin
  ClearList(ClassList);
end;

procedure TTopoView.ClearMarkers;
begin
  ClearList(MarkerList);
end;

{ TTopoViewVBO }

procedure TTopoViewVBO.InitGL;
begin
  inherited;
  glDepthFunc(GL_LESS);
  ReadExtensions;
  UseVBuffer:= GL_ARB_vertex_buffer_object;
  if UseVBuffer then begin
    // Create VBO
    glGenBuffersARB(1, @VBuffer);
    // Set VBO's data
    glBindBufferARB(GL_ARRAY_BUFFER_ARB, VBuffer);
    glEnableClientState(GL_VERTEX_ARRAY);
  end;
end;

procedure TTopoViewVBO.MakeVertex(Size: integer);
const
 VertPos : array[0..3] of array[0..1] of Byte = ((0,0),(0,1),(1,1),(1,0));
var
   x,z,i      : LongInt;
   Vertex     : array[0..3] of record X,Y,Z:single; end;
   rs         : single;
   VPointer   : ^TVBOVertex;
begin
  if not UseVBuffer then begin inherited; exit; end;
  // Bind VBO
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, VBuffer);
  // Set VBOs size
  glBufferDataARB(GL_ARRAY_BUFFER_ARB, Sqr(FResolution-1)*4*SizeOf(TVBOVertex), nil, GL_STATIC_DRAW_ARB);
  VBOPointer := glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB);
  VBOSize    := 0;
  FSize:= Size;

  rs:= FRawres/FResolution;
  DoProgress(0);
  // Write the vertices of the heightmap directly to VRAM
  for x := 0 to FResolution-2 do begin
    for z := 0 to FResolution-2 do begin
      for i := 0 to 3 do begin
        Vertex[i].X := (X + VertPos[I][0])/FResolution*Size-Size/2;
        Vertex[i].Z := (Z + VertPos[I][1])/FResolution*Size-Size/2;
        Vertex[i].Y := GetHeight(trunc((X + VertPos[I][0])*rs),trunc((Z + VertPos[I][1])*rs));

        if Vertex[i].Y=iNoData then
          Vertex[i].Y:= FLowest-1;

        VPointer    := VBOPointer;
        VPointer^.X := Vertex[i].X;
        VPointer^.Y := Vertex[i].Y;
        VPointer^.Z := Vertex[i].Z;
        VPointer^.S := (Vertex[i].X-Size/2)/Size;
        VPointer^.T := (-Vertex[i].Z-Size/2)/Size;

        inc(Integer(VBOPointer), SizeOf(TVBOVertex));
        inc(VBOSize);
      end;
    end;
    DoProgress(x/FResolution-2);
  end;
  // We need to unmap our VBO before we can draw it
  glUnMapBufferARB(GL_ARRAY_BUFFER_ARB);
end;


procedure TTopoViewVBO.DoRealDraw;
begin
  if not UseVBuffer then begin inherited; exit; end;
  glBindBufferARB(GL_ARRAY_BUFFER_ARB, VBuffer);
  glInterleavedArrays(GL_T2F_V3F, SizeOf(TVBOVertex), nil);
  glDrawArrays(GL_QUADS, 0, VBOSize);
end;



{ TTopoViewCLOD }

function MaxArr(const Values : Array of Single): Single;
////////////////////////////////////////////////////////////////////////////////
// Max
//------------------------------------------------------------------------------
// Whlt den grten Wert in einem Array
//------------------------------------------------------------------------------
// Parameter: Values: Array der Werte
// Rckgabe : Grter Wert
////////////////////////////////////////////////////////////////////////////////
var
  I : Integer;
begin
  Result := 0;
  for I := Low(Values) to High(Values) do
    if (Values[I] > Result) then Result := Values[I]
end;

function MaxByte(const Values : Array of Byte): Byte;
////////////////////////////////////////////////////////////////////////////////
// MaxByte
//------------------------------------------------------------------------------
// Whlt den grten Wert in einem Array of Byte
//------------------------------------------------------------------------------
// Parameter: Values: Array der Werte
// Rckgabe : Grter Wert
////////////////////////////////////////////////////////////////////////////////
var
  I : Integer;
begin
  Result := 0;
  for I := Low(Values) to High(Values) do
    if (Values[I] > Result) then Result := Values[I]
end;


function Max2Byte(Values : TQuadTreeMatrix): Byte;
////////////////////////////////////////////////////////////////////////////////
// Max2Byte
//------------------------------------------------------------------------------
// Whlt den grten Wert in einem Array of array of Byte
//------------------------------------------------------------------------------
// Parameter: Values: Array der Werte
// Rckgabe : Grter Wert
////////////////////////////////////////////////////////////////////////////////
var
  I,J : Integer;
begin
  Result := 0;
  for J := Low(Values) to High(Values) do
    for I := Low(Values[J]) to High(Values[J]) do
      if (Values[I,J] > Result) then Result := Values[I,J];
end;


procedure TTopoViewCLOD.QTPosToVertexPos(var V: TVector3f);
begin
  V[cX] := V[cX] / (QuadTreeLength - 1) * FSize - FSize/2;
  V[cZ] := V[cZ] / (QuadTreeLength - 1) * FSize - FSize/2;
end;

procedure TTopoViewCLOD.QTPosToRawPos(var V: TVector3f);
begin
  V[cX] :=V[cX] / (QuadTreeLength - 1) * FRawRes;
  V[cZ] :=V[cZ] / (QuadTreeLength - 1) * FRawRes;
end;

function TTopoViewCLOD.GetHeightMapHeightQT(const X,Z : Single): Single;
var
  v: TVector3f;
begin
  v[cX]:= X;
  v[cZ]:= Z;
  QTPosToRawPos(v);
  Result := getheight(trunc(v[cX]),trunc(v[cZ]))/Height_Scale;
end;

procedure TTopoViewCLOD.CreateQuadTree;

  procedure CheckMatrixCell(const X, Z, Size : Longint);
  ////////////////////////////////////////////////////////////////////////////////
  // CheckMatrixCell
  //------------------------------------------------------------------------------
  // berprft ob eine Quadtreenode gesetzt werden soll oder nicht und veranlasst
  // das gleiche fr seine Kinder
  //------------------------------------------------------------------------------
  // Parameter: X,Z: Koordinaten in der Matrix, der zu berprfenden Node
  // Rckgabe : ---
  ////////////////////////////////////////////////////////////////////////////////

  var
    CellCenterPos: TVector3f;
    HSize : LongInt;
  begin
    CellCenterPos[cX] := X;
    CellCenterPos[cH] := 0;
    CellCenterPos[cZ] := Z;


    if ( C/
         (Size*2* Max(kleinC*D2Matrix[X,Z], 1.0))
       )<1 then
    begin
      QuadMatrix[X,Z] := 1;
      if Size > 1 then
      begin
        HSize := Size div 2;
        CheckMatrixCell(X+ HSize, Z + HSize, HSize);
        CheckMatrixCell(X+ HSize, Z - HSize, HSize);
        CheckMatrixCell(X- HSize, Z + HSize, HSize);
        CheckMatrixCell(X- HSize, Z - HSize, HSize)
      end
    end
  end; {CheckMatrixCell}

begin
  //Matrix entleeren
  ZeroQT(QuadMatrix);
  //Quad Matrix erstellen
  CheckMatrixCell((QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2);
end;


procedure TTopoViewCLOD.RenderQuadTree;

  procedure RenderQuadTreeCell(const X,Z, Size : LongInt);
  ////////////////////////////////////////////////////////////////////////////////
  // RenderQuadTreeCell
  //------------------------------------------------------------------------------
  // Rendert eine Quadtreenode an der Stelle X,Z und veranlasst das gleiche fr
  // ihre Kinder -- knnte insgesamt stark verbessert werden
  //------------------------------------------------------------------------------
  // Parameter: X,Z: Koordinaten der Node
  // Rckgabe : ---
  ////////////////////////////////////////////////////////////////////////////////

    function GetNode(const X,Z : LongInt): Byte;
    ////////////////////////////////////////////////////////////////////////////////
    // GetNode
    //------------------------------------------------------------------------------
    // Holt den Zustand einer Benachbarten Node
    //------------------------------------------------------------------------------
    // Parameter: X,Z: Koordinaten
    // Rckgabe : Zusand der gewnschten Node, oder 1, falls die Node nicht
    //            existiert
    ////////////////////////////////////////////////////////////////////////////////
    begin
      if (X<0) or (Z<0) or (X > QuadTreeLength -1) or
        (Z > QuadTreeLength -1) then
        Result := 1
      else
        Result := QuadMatrix[X,Z]
    end; (*GetNode*)

    function GetHeightVertex(const X,Z: LongInt): TVector3f;
    ////////////////////////////////////////////////////////////////////////////////
    // GetHeightVertex
    //------------------------------------------------------------------------------
    // Erzeugt ein Vertex fr einen Bestimmten Punkt in der Matrix. Stimmt die
    // Matrixauflsung mir der angezeigten gre berein oder wird glScalex
    // verwendet, kann diese Funktion weggelassen werden - Performance
    //------------------------------------------------------------------------------
    // Parameter: X,Z: Koordianten
    // Rckgabe : Vertex der Koordinaten
    ////////////////////////////////////////////////////////////////////////////////
    begin
      Result[cX] := X;
      Result[cZ] := Z;
      QTPosToVertexPos(Result);
      Result[cH] := GetHeightMapHeight(Result[cX],Result[cZ]);
    end; (*GetHeightVertex*)


  const
    Vertieces : Array[1..9] of Array[0..1] of ShortInt =
      ((+1, +1),(+1, +0), (+1, -1), (+0, -1), (-1, -1), (-1, +0), (-1, +1), (+0,+1), (+1, +1));

    ChildPoss : Array[0..3] of Array[0..1] of ShortInt =
      ((+1, +1),(+1, -1),(-1, -1),(-1,+1));

    NeighborPoss : Array[0..3] of Array[0..1] of ShortInt =
      ((1,0),(0,-1),(-1,0),(0,1));

  var
    HSize : LongInt; //Hlfte der eigenen Seitenlnge
    I : Integer;

    Childs : Array[0..4] of Byte;
    Neighbours : Array[0..3] of Byte;

    MidVertex : TVector3f;

    procedure DrawPolygon(const V1, V2 : LongInt);
      procedure DrawVertex(var vertex:TVector3f);
      var t: TVector3f;
      begin
        t[0]:= (Vertex[cX]-FSize/2)/FSize;
        t[1]:= (-Vertex[cZ]-FSize/2)/FSize;

//        glTexCoord2f(t[0],t[1]);
//        glVertex3fv(@Vertex[0]);

        SetLength(TexCoordArray, Length(TexCoordArray)+1);
        TexCoordArray[High(TexCoordArray)][0]:= t[0];
        TexCoordArray[High(TexCoordArray)][1]:= t[1];
        SetLength(VertexArray, Length(VertexArray)+1);
        VertexArray[High(VertexArray)][0] := Vertex[0];
        VertexArray[High(VertexArray)][1] := Vertex[1];
        VertexArray[High(VertexArray)][2] := Vertex[2];
      end;

    var
      StartVertex, EndVertex, Normal : TVector3f;
      XStart, XEnd, ZStart, ZEnd : LongInt;
    begin
      if (V1 mod 2 = 1) and (Childs[V1 div 2] > 0) then Exit;
      if (V2 mod 2 = 1) and (Childs[V2 div 2] > 0) then Exit;

      XStart := X + Size * Vertieces[V1][0];
      ZStart := Z + Size * Vertieces[V1][1];
      XEnd := X + Size * Vertieces[V2][0];
      ZEnd := Z + Size * Vertieces[V2][1];
      StartVertex := GetHeightVertex(XStart,ZStart);
      EndVertex := GetHeightVertex(XEnd, ZEnd);

      Normal := VectorCrossProduct(VectorSubtract(MidVertex, StartVertex),
                 VectorSubtract(StartVertex, EndVertex));
      NormalizeVector(Normal);
      glNormal3fv(@Normal[0]);
      glBegin(GL_TRIANGLES);
        DrawVertex(MidVertex);
        DrawVertex(StartVertex);
        DrawVertex(EndVertex);
      glEnd()
    end;

  var
    StartVert : Integer;
  begin
    //Wenn die QuadMatrix an unserer Stelle 0 ist, wird nix angezeigt
    if QuadMatrix[X,Z] = 0 then Exit;

    HSize := Size div 2;
    MidVertex := GetHeightVertex(X,Z);

    FillChar(Childs[0], SizeOf(Childs), 0);
    if Size > 1 then
      for i := 0 to 3 do
        Childs[i] := QuadMatrix[X + HSize * ChildPoss[i][0],
                             Z + HSize * ChildPoss[i][1]];
    Childs[4] := Childs[0];

    for i := 0 to 3 do
      Neighbours[i] := GetNode(X + 2*Size * NeighborPoss[i][0],
                           Z + 2*Size * NeighborPoss[i][1]);

    StartVert := 1;
    for i := 2 to 9 do
    begin
      if i mod 2 = 0 then
      begin
        if (Neighbours[(i div 2)-1])> 0 then
        begin
          DrawPolygon(StartVert, I);
          StartVert := I
        end
      end
      else
      begin
        DrawPolygon(StartVert, I);
        StartVert := I
      end
    end;

    if Size > 1 then
    begin
      HSize := Size div 2;
      RenderQuadTreeCell(X+ HSize, Z + HSize, HSize);
      RenderQuadTreeCell(X+ HSize, Z - HSize, HSize);
      RenderQuadTreeCell(X- HSize, Z + HSize, HSize);
      RenderQuadTreeCell(X- HSize, Z - HSize, HSize)
    end
  end; (*RenderQuadTreeCell*)

begin
  //Quad Matrix rendern
  RenderQuadTreeCell((QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2);
end;

procedure TTopoViewCLOD.DoD2CrackAvoid;
////////////////////////////////////////////////////////////////////////////////
// DoD2CrackAvoid
//------------------------------------------------------------------------------
// Errechnet die D2 Werte und verhinder Cracks im Mesh
//------------------------------------------------------------------------------
// Parameter: ---
// Rckgabe : ---
////////////////////////////////////////////////////////////////////////////////

var
  K : Single;

  procedure CalcD2Value(X, Z, Size : Longint);
  ////////////////////////////////////////////////////////////////////////////////
  // CalcD2Calue
  //------------------------------------------------------------------------------
  // Erzeugt den D2 Wert fr eine bestimmte Node, und veranlasst das gleiche fr
  // deren Kinder
  //------------------------------------------------------------------------------
  // Parameter: X,Z: koordinaten der Node
  // Rckgabe : ---
  ////////////////////////////////////////////////////////////////////////////////

  var
    HSize : LongInt; //Hlfte der eigenen Seitenlnge
    I : LongInt;
    EdgeHeights : Array[1..4] of Single; //Hhe der Ecken
    ExpectedHeight : Single;             //Erwartete hhe zw 2 Punkten
    RealHeight : Single;                 //Tatschliche Hhe

    DHValues : Array[1..6] of Single;    //DHWerte

  const
    //Ort der Ecken des Quads
    Edges : Array[1..4] of Array[0..1] of ShortInt = ((+1, +1),(+1, -1),(-1,-1),(-1,+1));
    //Jeweils die 2 Ecken, zwischen denen ein DHWert berechnet wird
    DHLines : Array[1..6] of Array[0..1] of Byte =
      ((1,2),(2,3),(3,4),(4,1),(1,3),(2,4));

    //Positionen der DHWerte selber
    DHPositions : Array[1..6] of Array[0..1] of ShortInt =
      ((1,0),(0,-1),(-1,0),(0,1),(0,0),(0,0));

  begin
    //Hhen der 4 Ecken
    for I := 1 to 4 do
      EdgeHeights[I] := GetHeightMapHeightQT(X + Size * Edges[i,0],
                          Z + Size * Edges[i,1]);

    //DHWerte berechen
    for i := 1 to 6 do
    begin
      //Erwartete Hhe zwischen 2 Punkten
      ExpectedHeight := (EdgeHeights[DHLines[i,0]] + EdgeHeights[DHLines[i,1]])/2;
      //Tatschlich vorliegende Hhe
      RealHeight := GetHeightMapHeightQT(
        X+ DHPositions[i,0] * Size,
        Z + DHPositions[i,1] * Size);
      //DH Wert ist |absoluten Fehlers|
      DHValues[I] := Abs(ExpectedHeight - RealHeight);
    end;

    //D2Value finden und setzen
    D2Matrix[X,Z] := Trunc(1/(2*Size) * MaxArr(DHValues));

    //D2Values fr die 4 Kinder
    if Size > 1 then
    begin
      HSize := Size div 2;
      CalcD2Value(X+ HSize, Z + HSize, HSize);
      CalcD2Value(X+ HSize, Z - HSize, HSize);
      CalcD2Value(X- HSize, Z + HSize, HSize);
      CalcD2Value(X- HSize, Z - HSize, HSize)
    end
  end; (*CalcD2Values*)

  procedure DoCrackAvoid(X, Z, Size, SizeToCheck : Longint);
  ////////////////////////////////////////////////////////////////////////////////
  // DoCrackAvoid
  //------------------------------------------------------------------------------
  // Modifiziert die D2 Werte einer Node in der Art, dass Cracks verhindert werden
  //------------------------------------------------------------------------------
  // Parameter: X,Z: koordinaten der Node; Size: Radius der Node;
  //            SizeToCheck: Radius der Nodes die bearbeitet werden
  // Rckgabe : ---
  ////////////////////////////////////////////////////////////////////////////////

    function GetNode(X,Z : LongInt): Byte;
    ////////////////////////////////////////////////////////////////////////////////
    // GetNode
    //------------------------------------------------------------------------------
    // Holt den D2 Wert einer Node
    //------------------------------------------------------------------------------
    // Parameter: X,Z: Koordinaten der Node
    // Rckgabe : D2 Wert der Node, bzw. 0, wenn die Node nicht existiert
    ////////////////////////////////////////////////////////////////////////////////

    begin
      if (X<0) or (Z<0) or (X > QuadTreeLength -1) or
        (Z > QuadTreeLength -1) then
        Result := 0
      else
        Result := D2Matrix[X,Z]
    end; (*GetNode*)

  const
    Neighbours : Array[0..8] of Array[0..1] of ShortInt =
      ((+0,+0), (+3,+1),(+3,-1),(+1,-3),(-1,-3),(-3,-1),(-3,+1),(-1,+3),(+1,+3));
  var
    HSize : LongInt; //Hlfte der eigenen Seitenlnge
    I : LongInt;
    D2Values : Array[0..8] of Byte;    //DHWerte

  begin
    //D2Values fr die 4 Kinder
    Assert(SizeToCheck mod 2 = 0);
    Assert(Size > 1);
    if Size > SizeToCheck then
    begin
      HSize := Size div 2;
      DoCrackAvoid(X+ HSize, Z + HSize, HSize,SizeToCheck);
      DoCrackAvoid(X+ HSize, Z - HSize, HSize,SizeToCheck);
      DoCrackAvoid(X- HSize, Z + HSize, HSize,SizeToCheck);
      DoCrackAvoid(X- HSize, Z - HSize, HSize,SizeToCheck)
    end
    else
    begin
      Assert(SizeToCheck = Size);
      HSize := Size div 2;
      D2Values[0] := D2Matrix[X,Z];
      for i := 1 to 8 do
        D2Values[i] := Round(GetNode(X+ HSize * Neighbours[i][0],
                               Z+ HSize * Neighbours[i][1]) * K);

      D2Matrix[X,Z] := MaxByte(D2Values)
    end
  end; (*DoCrackAvoid*)

var
  I : Integer;

begin
  //Zu K wie es im Tutorial steht, wird ein kleiner Wert hinzuaddiert,
  //um zu verhindern dass Cracks auftauchen, da die D2 Werte ja auf
  //Bytes komprimiert sind.
  K := C/(C-1)+0.15;

  CalcD2Value((QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2,
    (QuadTreeLength-1) div 2);

  I := 2;
  DoProgress(0);
  while I < (QuadTreeLength - 1) div 2 do
  begin
    DoCrackAvoid((QuadTreeLength-1) div 2, (QuadTreeLength-1) div 2,
      (QuadTreeLength-1) div 2, I);
    I := I * 2;
    DoProgress((I-2)/((QuadTreeLength - 1) div 2));
  end;
//  Assert(not Max2Byte(D2Matrix)=0, 'Error in D2MAtrix. All values Zero');
  DoProgress(1);
end;

procedure TTopoViewCLOD.InitGL;
begin
  inherited;
  C:= 15.0;
  kleinC:= 1.0;
  //Kamera Initialisieren
  KameraPosition[cX] := 0;
  KameraPosition[cH] := 0;
  KameraPosition[cZ] := 0;   
end;

procedure TTopoViewCLOD.DoRealDraw;
begin
  //                      ( % of shown              *  size of qt      )
  KameraPosition[cX] := ((FSize/2-CenterX)/FSize) * (QuadTreeLength-1);
  KameraPosition[cZ] := ((FSize/2-CenterZ)/FSize) * (QuadTreeLength-1);
  KameraPosition[cH] := GetHeightMapHeightQT(KameraPosition[cX],KameraPosition[cZ]);


  glVertexPointer(3, GL_FLOAT, 12, VertexArray);
  glTexCoordPointer(2, GL_FLOAT, 8, TexCoordArray);
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glDrawArrays(GL_TRIANGLES, 0, Length(VertexArray));    
end;


function myCeil(X: Extended): Integer;
begin
  Result := Geometry.Trunc(X);       // Some strange error in System.Trunc(), so use
  if Frac(X) > 0 then                // another one from Geometry.pas
    Inc(Result);
end;

procedure TTopoViewCLOD.MakeVertex(Size: integer);
var ex:double;
begin
  FSize:= Size;
  ex:= log2(FSize);
  ex:= myCeil(ex)+2;
  QuadTreeLength:= Min(trunc(IntPower(2,Geometry.Trunc(ex))),2048)+1;
  SetLength(QuadMatrix,QuadTreeLength,QuadTreeLength);
  SetLength(D2Matrix,QuadTreeLength,QuadTreeLength);
  QTSize:= QuadTreeLength*QuadTreeLength*Sizeof(TQuadTreeEntry);
  DoD2CrackAvoid;

  SetLength(TexCoordArray,0);
  SetLength(VertexArray, 0);
  CreateQuadTree;
  RenderQuadTree;
end;



procedure TTopoViewCLOD.ZeroQT(var qt: TQuadTreeMatrix);
var i,j:integer;
begin
  for i:= 0 to QuadTreeLength-1 do
    for j:= 0 to QuadTreeLength-1 do
      qt[i,j]:= 0;
end;

end.
